<?php

/**
 * @file
 * Stub file for bootstrap_pager().
 */

/* @noinspection PhpDocMissingThrowsInspection */

/**
 * Returns HTML for a query pager.
 *
 * Menu callbacks that display paged query results should call theme('pager') to
 * retrieve a pager control so that users can view other results. Format a list
 * of nearby pages with additional query results.
 *
 * @param array $variables
 *   An associative array containing:
 *   - tags: An array of labels for the controls in the pager.
 *   - element: An optional integer to distinguish between multiple pagers on
 *     one page.
 *   - parameters: An associative array of query string parameters to append to
 *     the pager links.
 *   - quantity: The number of pages in the list.
 *
 * @return string
 *   The constructed HTML.
 *
 * @see theme_pager()
 *
 * @ingroup theme_functions
 */
function bootstrap_pager(array $variables)
{
    $output = "";
    $items = array();
    $tags = $variables['tags'];
    $element = $variables['element'];
    $parameters = $variables['parameters'];
    $quantity = $variables['quantity'];

    global $pager_page_array, $pager_total;

    // Calculate various markers within this pager piece:
    // Middle is used to "center" pages around the current page.
    $pager_middle = ceil($quantity / 2);
    // Current is the page we are currently paged to.
    $pager_current = $pager_page_array[$element] + 1;
    // First is the first page listed by this pager piece (re quantity).
    $pager_first = $pager_current - $pager_middle + 1;
    // Last is the last page listed by this pager piece (re quantity).
    $pager_last = $pager_current + $quantity - $pager_middle;
    // Max is the maximum page number.
    $pager_max = $pager_total[$element];

    // Prepare for generation loop.
    $i = $pager_first;
    if ($pager_last > $pager_max) {
        // Adjust "center" if at end of query.
        $i = $i + ($pager_max - $pager_last);
        $pager_last = $pager_max;
    }
    if ($i <= 0) {
        // Adjust "center" if at start of query.
        $pager_last = $pager_last + (1 - $i);
        $i = 1;
    }

    // End of generation loop preparation.
    /* @noinspection PhpUnhandledExceptionInspection */
    $li_first = partners_pager_first(array(
        'text' => (isset($tags[0]) ? $tags[0] : t('first')),
        'element' => $element,
        'parameters' => $parameters,
    ));
    /* @noinspection PhpUnhandledExceptionInspection */
    $li_previous = partners_pager_previous(array(
        'text' => (isset($tags[1]) ? $tags[1] : t('previous')),
        'element' => $element,
        'interval' => 1,
        'parameters' => $parameters,
    ));
    /* @noinspection PhpUnhandledExceptionInspection */
    $li_next = partners_pager_next(array(
        'text' => (isset($tags[3]) ? $tags[3] : t('next')),
        'element' => $element,
        'interval' => 1,
        'parameters' => $parameters,
    ));
    /* @noinspection PhpUnhandledExceptionInspection */
    $li_last = partners_pager_last(array(
        'text' => (isset($tags[4]) ? $tags[4] : t('last')),
        'element' => $element,
        'parameters' => $parameters,
    ));
    if ($pager_total[$element] > 1) {

        // Only show "first" link if set on components' theme setting.
        if ($li_first && bootstrap_setting('pager_first_and_last')) {
            $items[] = array(
                'class' => array('pager-first'),
                'data' => $li_first,
            );
        }
        if ($li_previous) {
            $items[] = array(
                'class' => array('prev'),
                'data' => $li_previous,
            );
        }
        // When there is more than one page, create the pager list.
        if ($i != $pager_max) {
            if ($i > 1) {
                $items[] = array(
                    'class' => array('pager-ellipsis', 'disabled'),
                    'data' => '<span>…</span>',
                );
            }
            // Now generate the actual pager piece.
            for (; $i <= $pager_last && $i <= $pager_max; $i++) {
                if ($i < $pager_current) {
                    /* @noinspection PhpUnhandledExceptionInspection */
                    $items[] = array(
                        'class' => array('page-item'),
                        'data' => partners_pager_previous( array(
                            'text' => $i,
                            'element' => $element,
                            'interval' => ($pager_current - $i),
                            'parameters' => $parameters,
                        )),
                    );
                }
                if ($i == $pager_current) {
                    $items[] = array(
                        // Add the active class.
                        'class' => array('active'),
                        'data' => "<span class='page-link text-white bg-primary'>$i</span>",
                    );
                }
                if ($i > $pager_current) {
                    /* @noinspection PhpUnhandledExceptionInspection */
                    $items[] = array(
                        'data' => partners_pager_next(array(
                            'text' => $i,
                            'element' => $element,
                            'interval' => ($i - $pager_current),
                            'parameters' => $parameters,
                        )),
                    );
                }
            }
            if ($i < $pager_max) {
                $items[] = array(
                    'class' => array('pager-ellipsis', 'disabled'),
                    'data' => '<span>…</span>',
                );
            }
        }
        // End generation.
        if ($li_next) {
            $items[] = array(
                'class' => array('next'),
                'data' => $li_next,
            );
        }
        // // Only show "last" link if set on components' theme setting.
        if ($li_last && bootstrap_setting('pager_first_and_last')) {
            $items[] = array(
                'class' => array('pager-last'),
                'data' => $li_last,
            );
        }

        $build = array(
            '#theme_wrappers' => array('container__pager'),
            '#attributes' => array(
                'class' => array(
                    'text-center'
                ),
            ),
            'pager' => array(
                '#theme' => 'item_list',
                '#items' => $items,
                '#attributes' => array(
                    'class' => array('pagination', 'navigation'),
                ),
            ),
        );
        return drupal_render($build);
    }
    return $output;
}


function partners_pager_link_custom($variables)
{
    $text = $variables['text'];
    $page_new = $variables['page_new'];
    $element = $variables['element'];
    $parameters = $variables['parameters'];
    $attributes =[];
    $page = isset($_GET['page']) ? $_GET['page'] : '';
    if ($new_page = implode(',', pager_load_array($page_new[$element], $element, explode(',', $page)))) {
        $parameters['page'] = $new_page;
    }
    $query = array();
    if (count($parameters)) {
        $query = drupal_get_query_parameters($parameters, array());
    }
    if ($query_pager = pager_get_query_parameters()) {
        $query = array_merge($query, $query_pager);
    }

    // Set each pager link title
    if (!isset($attributes['title'])) {
        static $titles = NULL;
        if (!isset($titles)) {
            $titles = array(
                t('« first') => t('Go to first page'),
                t('‹ previous') => t('Go to previous page'),
                t('next ›') => t('Go to next page'),
                t('last »') => t('Go to last page'),
            );
        }
        if (isset($titles[$text])) {
            $attributes['title'] = $titles[$text];
        } elseif (is_numeric($text)) {
            $attributes['title'] = t('Go to page @number', array(
                '@number' => $text,
            ));
        }
    }

    // @todo l() cannot be used here, since it adds an 'active' class based on the
    //   path only (which is always the current path for pager links). Apparently,
    //   none of the pager links is active at any time - but it should still be
    //   possible to use l() here.
    // @see http://drupal.org/node/1410574
    $attributes['href'] = url($_GET['q'], array(
        'query' => $query,
    ));

    $attributes['class'] = "page-link";
    return '<a' . drupal_attributes($attributes) . '>' . check_plain($text) . '</a>';
}


function partners_pager_last_custom($variables)
{
    $text = $variables['text'];
    $element = $variables['element'];
    $parameters = $variables['parameters'];
    global $pager_page_array, $pager_total;
    $output = '';

    // If we are anywhere but the last page
    if ($pager_page_array[$element] < $pager_total[$element] - 1) {
        $output = partners_pager_link_custom(array(
            'text' => $text,
            'page_new' => pager_load_array($pager_total[$element] - 1, $element, $pager_page_array),
            'element' => $element,
            'parameters' => $parameters,
        ));
    }
    return $output;
}


function partners_pager_first($variables)
{
    $text = $variables['text'];
    $element = $variables['element'];
    $parameters = $variables['parameters'];
    global $pager_page_array;
    $output = '';

    // If we are anywhere but the first page
    if ($pager_page_array[$element] > 0) {
        $output = partners_pager_link_custom(array(
            'text' => $text,
            'page_new' => pager_load_array(0, $element, $pager_page_array),
            'element' => $element,
            'parameters' => $parameters,
        ));
    }
    return $output;
}

function partners_pager_previous($variables)
{
    $text = $variables['text'];
    $element = $variables['element'];
    $interval = $variables['interval'];
    $parameters = $variables['parameters'];
    global $pager_page_array;
    $output = '';

    // If we are anywhere but the first page
    if ($pager_page_array[$element] > 0) {
        $page_new = pager_load_array($pager_page_array[$element] - $interval, $element, $pager_page_array);

        // If the previous page is the first page, mark the link as such.
        if ($page_new[$element] == 0) {
            $output = partners_pager_first(array(
                'text' => $text,
                'element' => $element,
                'parameters' => $parameters,
            ));
        } else {
            $output = partners_pager_link_custom(array(
                'text' => $text,
                'page_new' => $page_new,
                'element' => $element,
                'parameters' => $parameters,
            ));
        }
    }
    return $output;
}

function partners_pager_next($variables)
{
    $text = $variables['text'];
    $element = $variables['element'];
    $interval = $variables['interval'];
    $parameters = $variables['parameters'];
    global $pager_page_array, $pager_total;
    $output = '';

    // If we are anywhere but the last page
    if ($pager_page_array[$element] < $pager_total[$element] - 1) {
        $page_new = pager_load_array($pager_page_array[$element] + $interval, $element, $pager_page_array);

        // If the next page is the last page, mark the link as such.
        if ($page_new[$element] == $pager_total[$element] - 1) {
            $output = partners_pager_last_custom(array(
                'text' => $text,
                'element' => $element,
                'parameters' => $parameters,
            ));
        } else {
            $output = partners_pager_link_custom(array(
                'text' => $text,
                'page_new' => $page_new,
                'element' => $element,
                'parameters' => $parameters,
            ));
        }
    }
    return $output;
}

function partners_pager_last($variables)
{
    $text = $variables['text'];
    $element = $variables['element'];
    $parameters = $variables['parameters'];
    global $pager_page_array, $pager_total;
    $output = '';

    // If we are anywhere but the last page
    if ($pager_page_array[$element] < $pager_total[$element] - 1) {
        $output = partners_pager_link_custom(array(
            'text' => $text,
            'page_new' => pager_load_array($pager_total[$element] - 1, $element, $pager_page_array),
            'element' => $element,
            'parameters' => $parameters,
        ));
    }
    return $output;
}