import $ from 'jquery';
import iziToast from 'izitoast';
import Cleave from 'cleave.js';
import numbro from 'numbro';
import moment from 'moment';
import PartnersModal from '../utilities/js/partnersModal';
import '../utilities/js/partnersBtn';
import { axiosInstance as axios, urls } from './config/config';
import '../../plugin/js/jquery.form-validator';
import '../../plugin/js/partnersfilter';
import '../../plugin/css/partnersfilter.css';
import 'bootstrap-datepicker';
import '../../plugin/css/bootstrap-datepicker3.css';
import DateRangePicker from '../utilities/js/dateRangePicker';
import loader from '../utilities/js/loader';

/**
 * Función para mostrar un modal con un formulario para capturar un vehículo.
 */
const vistaNuevoVehiculo = () => {
  const Modal = new PartnersModal(
    {
      title: 'Dar de Alta Vehículo',
      modalSize: 'modal-lg',
      modalId: 'nuevo-vehiculo-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}nuevo_vehiculo`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nuevo_vehiculo_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para capturar un vehículo en la base de datos, se validan todos lo campos obligatorios.
 * @param e
 * @param reporteador
 */
const capturarVehiculo = (e, reporteador) => {
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const datos = $(e.currentTarget).serializeArray();
  const datosEnvio = {};
  for (let i = 0; i < datos.length; i += 1) {
    datosEnvio[datos[i].name] = datos[i].value;
  }

  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_vehiculo`, datosEnvio)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nuevo-vehiculo-modal').modal('hide');
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para mostrar un modal con un formulario para capturar un gasto.
 */
const vistaNuevoGasto = () => {
  const Modal = new PartnersModal(
    {
      title: 'Gasto / Reparación',
      modalSize: 'modal-lg',
      modalId: 'nuevo-gasto-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}nuevo_gasto`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nuevo_gasto_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });

        // eslint-disable-next-line no-new
        new Cleave('#nuevo_gasto_form input[name="cantidad"]', {
          numeral: true,
          numeralThousandsGroupStyle: 'thousand',
        });

        // eslint-disable-next-line no-new
        new Cleave('#nuevo_gasto_form input[name="km"]', {
          numeral: true,
          numeralThousandsGroupStyle: 'thousand',
        });

        $('#nuevo_gasto_form .input-group.date').datepicker({
          format: 'dd/mm/yyyy',
          todayBtn: 'linked',
          language: 'es',
          daysOfWeekDisabled: '0',
          autoclose: true,
          todayHighlight: true,
          orientation: 'auto',
          container: '.input-group.date',
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para capturar un Gasto/Reparación de un vehículo.
 * @param e
 * @param reporteador
 */
const capturarGasto = (e, reporteador) => {
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const datos = $(e.currentTarget).serializeArray();
  const datosEnvio = {};
  for (let i = 0; i < datos.length; i += 1) {
    datosEnvio[datos[i].name] = datos[i].value;
  }

  // Validar fecha
  const fechaGasto = $('#nuevo_gasto_form .input-group.date').datepicker('getDate');
  if (!fechaGasto) {
    iziToast.error({
      title: 'Error',
      message: 'Selecciona la fecha del Gasto.',
    });
    return;
  }

  datosEnvio.fecha_gasto = moment(fechaGasto).unix();

  // Cantidad
  datosEnvio.cantidad = numbro(datosEnvio.cantidad).value();
  datosEnvio.km = numbro(datosEnvio.km).value();

  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_gasto`, datosEnvio)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nuevo-gasto-modal').modal('hide');
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para mostrar un modal con un formulario para capturar un seguro
 */
const vistaNuevoSeguro = () => {
  const Modal = new PartnersModal(
    {
      title: 'Seguro',
      modalSize: 'modal-lg',
      modalId: 'nuevo-seguro-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}nuevo_seguro`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nuevo_seguro_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });

        // eslint-disable-next-line no-new
        new Cleave('#nuevo_seguro_form input[name="precio"]', {
          numeral: true,
          numeralThousandsGroupStyle: 'thousand',
        });

        $('#nuevo_seguro_form .input-group.date').datepicker({
          format: 'dd/mm/yyyy',
          todayBtn: 'linked',
          language: 'es',
          daysOfWeekDisabled: '0',
          autoclose: true,
          todayHighlight: true,
          orientation: 'auto',
          container: '.input-group.date',
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para capturar un Seguro de un vehículo.
 * @param e
 * @param reporteador
 */
const capturarSeguro = (e, reporteador) => {
  // Como se va a subir una imágen, hay que usar el formData
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const formData = new FormData(e.currentTarget);

  // Validar fecha
  const fechaVencimiento = $('#nuevo_seguro_form .input-group.date').datepicker('getDate');
  if (!fechaVencimiento) {
    iziToast.error({
      title: 'Error',
      message: 'Selecciona la fecha de vencimiento.',
    });
    return;
  }

  formData.append('fecha_vencimiento', moment(fechaVencimiento).unix().toString());

  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_seguro`, formData)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nuevo-seguro-modal').modal('hide');
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para mostrar un modal con un formulario para capturar un kilometraje
 */
const vistaNuevoKm = () => {
  const Modal = new PartnersModal(
    {
      title: 'Kilometraje',
      modalSize: 'modal-lg',
      modalId: 'nuevo-km-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}nuevo_km`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nuevo_km_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });

        // eslint-disable-next-line no-new
        new Cleave('#nuevo_km_form input[name="kilometros"]', {
          numeral: true,
          numeralThousandsGroupStyle: 'thousand',
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para capturar un Kilometraje de un vehículo.
 * @param e
 * @param reporteador
 * @param external
 */
const capturarKm = (e, reporteador, external = false) => {
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const datos = $(e.currentTarget).serializeArray();
  const datosEnvio = {};
  for (let i = 0; i < datos.length; i += 1) {
    datosEnvio[datos[i].name] = datos[i].value;
  }

  datosEnvio.kilometros = numbro(datosEnvio.kilometros).value();

  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_km`, datosEnvio)
    .then((res) => {
      if (res.data.status === 'success') {
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });

        // Si no es una llamada externa entonces trabajarlo normal
        if (!external) {
          $('#nuevo-km-modal').modal('hide');
          reporteador.forceUpdate();
        } else {
          // eslint-disable-next-line no-restricted-globals
          location.reload();
        }
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para mostrar un modal con un formulario para capturar una Afinación
 */
const vistaNuevaAfinacion = () => {
  const Modal = new PartnersModal(
    {
      title: 'Afinación',
      modalSize: 'modal-lg',
      modalId: 'nueva-afinacion-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}nueva_afinacion`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nueva_afinacion_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });

        // eslint-disable-next-line no-new
        new Cleave('#nueva_afinacion_form input[name="cantidad"]', {
          numeral: true,
          numeralThousandsGroupStyle: 'thousand',
        });

        // eslint-disable-next-line no-new
        new Cleave('#nueva_afinacion_form input[name="km"]', {
          numeral: true,
          numeralThousandsGroupStyle: 'thousand',
        });

        $('#nueva_afinacion_form .input-group.date').datepicker({
          format: 'dd/mm/yyyy',
          todayBtn: 'linked',
          language: 'es',
          daysOfWeekDisabled: '0',
          autoclose: true,
          todayHighlight: true,
          orientation: 'auto',
          container: '.input-group.date',
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para capturar una Afinación de un vehículo.
 * @param e
 * @param reporteador
 */
const capturarAfinacion = (e, reporteador) => {
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const datos = $(e.currentTarget).serializeArray();
  const datosEnvio = {};
  for (let i = 0; i < datos.length; i += 1) {
    datosEnvio[datos[i].name] = datos[i].value;
  }

  // Validar fecha
  const fechaAfinacion = $('#nueva_afinacion_form .input-group.date').datepicker('getDate');
  if (!fechaAfinacion) {
    iziToast.error({
      title: 'Error',
      message: 'Selecciona la fecha de la afinación.',
    });
    return;
  }

  datosEnvio.fecha_gasto = moment(fechaAfinacion).unix();

  datosEnvio.km = numbro(datosEnvio.km).value();
  datosEnvio.concepto = 'Afinación';

  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_afinacion`, datosEnvio)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nueva-afinacion-modal').modal('hide');
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para mostrar un modal con un formulario para capturar un refrendo
 */
const vistaNuevoRefrendo = () => {
  const Modal = new PartnersModal(
    {
      title: 'Refrendo',
      modalSize: 'modal-lg',
      modalId: 'nuevo-refrendo-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}nuevo_refrendo`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nuevo_refrendo_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });

        $('#nuevo_refrendo_form .input-group.date').datepicker({
          format: 'dd/mm/yyyy',
          todayBtn: 'linked',
          language: 'es',
          daysOfWeekDisabled: '0',
          autoclose: true,
          todayHighlight: true,
          orientation: 'auto',
          container: '.input-group.date',
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para capturar un Refrendo de un vehículo.
 * @param e
 * @param reporteador
 */
const capturarRefrendo = (e, reporteador) => {
  // Como se va a subir una imágen, hay que usar el formData
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const formData = new FormData(e.currentTarget);

  // Validar fecha
  const fechaRefrendo = $('#nuevo_refrendo_form .input-group.date').datepicker('getDate');
  if (!fechaRefrendo) {
    iziToast.error({
      title: 'Error',
      message: 'Selecciona la fecha.',
    });
    return;
  }

  formData.append('fecha_refrendo', moment(fechaRefrendo).unix().toString());

  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_refrendo`, formData)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nuevo-refrendo-modal').modal('hide');
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para abrir un archivo en particular en una pestaña nueva.
 * @param file Nombre del archivo
 * @param fileUrl Ruta de la carpeta en donde se encuentra el archivo
 */
const visualizarArchivo = (file, fileUrl) => {
  window.open(`${fileUrl}?file=${file}`);
};

/**
 * Función que copia el contenido de un elemento al portapapeles.
 * @param e
 */
const copyToClipboard = (e) => {
  const $temp = $('<input>');
  const $this = $(e.currentTarget);
  $('body').append($temp);
  $temp.val($(e.currentTarget).text()).select();
  document.execCommand('copy');
  $temp.remove();
  $this.tooltip({
    trigger: 'click',
    placement: 'top',
  });

  $this.tooltip('hide')
    .attr('data-original-title', 'Copiado!')
    .tooltip('show');
  setTimeout(() => {
    $this.tooltip('dispose');
  }, 1000);
};

/**
 * Función para generar una vista detallada de un vehículo.
 * @param e
 */
const vistaVehiculo = (e) => {
  const idVehiculo = $(e.currentTarget).parents('tr').attr('data-id-vehiculo');
  const Modal = new PartnersModal(
    {
      title: 'Vista Vehículo',
      modalSize: 'modal-lg',
      modalId: 'vista-vehiculo-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.post(`${urls.consultas}vista_vehiculo`, idVehiculo)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};


/**
 * Función para generar la tabla de la vista del kilometraje.
 * @param idVehiculo
 * @param start
 * @param end
 */
const reporteKilometraje = (idVehiculo, start, end) => {
  // Mostrar el loader
  const reporteadorKm = $('#reporteador_kilometraje');
  reporteadorKm.html(loader);

  axios.post(`${urls.consultas}reporte_kilometraje`, { idVehiculo, start, end })
    .then((res) => {
      if (res.data.status === 'success') {
        reporteadorKm.html(res.data.message.table);
        $('#reporteador_kilometraje_recorrido').html(res.data.message.km_recorridos);
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};


/**
 * Función para mostrar un modal con todos los kilometrajes de un vehículo.
 * @param e
 */
const vistaKilometraje = (e) => {
  const idVehiculo = $(e.currentTarget).parents('tr').attr('data-id-vehiculo');
  const nombreVehiculo = $(e.currentTarget).parents('tr').attr('data-nombre-vehiculo');
  const start = moment().startOf('month');
  const end = moment().endOf('month');
  const Modal = new PartnersModal(
    {
      title: `Kilometraje ${nombreVehiculo}`,
      modalSize: 'modal-lg',
      modalId: 'vista-kilometraje-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}vista_kilometraje`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);

        const picker = new DateRangePicker('#reportrange_kilometraje', start, end, {}, '.parent-calendar-kilometraje');
        reporteKilometraje(idVehiculo, start, end);
        $(picker.getElement()).on('apply.daterangepicker', (event, datePicker) => {
          picker.updateDates(datePicker.startDate, datePicker.endDate);
          reporteKilometraje(idVehiculo, datePicker.startDate, datePicker.endDate);
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para generar la tabla de la vista del Gasto Anual.
 * @param idVehiculo
 * @param start
 * @param end
 */
const reporteGastoAnual = (idVehiculo, start, end) => {
  // Mostrar el loader
  const reporteadorGastoAnual = $('#reporteador_gasto_anual');
  reporteadorGastoAnual.html(loader);

  axios.post(`${urls.consultas}reporte_gasto_anual`, { idVehiculo, start, end })
    .then((res) => {
      if (res.data.status === 'success') {
        reporteadorGastoAnual.html(res.data.message);
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para mostrar un modal con todos los gastos anuales de un vehículo.
 * @param e
 */
const vistaGastoAnual = (e) => {
  const idVehiculo = $(e.currentTarget).parents('tr').attr('data-id-vehiculo');
  const nombreVehiculo = $(e.currentTarget).parents('tr').attr('data-nombre-vehiculo');
  const start = moment().startOf('year');
  const end = moment().endOf('year');
  const Modal = new PartnersModal(
    {
      title: `Gasto Anual ${nombreVehiculo}`,
      forceWidth: 80,
      modalId: 'vista-gasto-anual-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}vista_gasto_anual`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);

        const picker = new DateRangePicker('#reportrange_gasto_anual', start, end, {}, '.parent-calendar-gasto-anual');
        reporteGastoAnual(idVehiculo, start, end);
        $(picker.getElement()).on('apply.daterangepicker', (event, datePicker) => {
          picker.updateDates(datePicker.startDate, datePicker.endDate);
          reporteGastoAnual(idVehiculo, datePicker.startDate, datePicker.endDate);
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};


/**
 * Función para mostrar un modal con un formulario para capturar una Entrada de Vales
 */
const vistaNuevaEntradaVales = () => {
  const Modal = new PartnersModal(
    {
      title: 'Entrada Vales',
      modalSize: 'modal-lg',
      modalId: 'nueva-entrada-vales-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}vista_entrada_vales`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nueva_entrada_vales_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });

        // eslint-disable-next-line no-new
        new Cleave('#nueva_entrada_vales_form input[name="precio_gasolina"]', {
          numeral: true,
          numeralThousandsGroupStyle: 'thousand',
        });

        $('.vale_monto').toArray().forEach((f) => {
          // eslint-disable-next-line no-new
          new Cleave(f, {
            numeral: true,
            numeralThousandsGroupStyle: 'thousand',
          });
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para capturar una Entrada de Vales.
 * @param e
 * @param reporteador
 */
const capturarEntradaVales = (e, reporteador) => {
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const datos = $(e.currentTarget).serializeArray();
  const datosEnvio = {};
  for (let i = 0; i < datos.length; i += 1) {
    datosEnvio[datos[i].name] = datos[i].value;
  }

  // Validar vales
  const listVales = [];
  let err = false;
  $('.tr_vale').toArray().forEach((tr) => {
    if (err) return;
    const valeFolio = $(tr).find('.vale_folio').val();
    const valeMonto = $(tr).find('.vale_monto').val();
    if (valeFolio !== '' && valeFolio !== undefined) {
      if (valeMonto === '' || valeMonto === undefined) {
        iziToast.error({
          title: 'Error',
          message: `El folio ${valeFolio} no tiene monto.`,
        });
        err = true;
      } else {
        listVales.push({ valeFolio, valeMonto: numbro(valeMonto).value() });
      }
    }
  });

  if (err) return;

  if (listVales.length < 1) {
    iziToast.error({
      title: 'Error',
      message: 'Ingresa al menos un vale.',
    });
    return;
  }

  datosEnvio.listVales = listVales;
  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_entrada_vales`, datosEnvio)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nueva-entrada-vales-modal').modal('hide');
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para mostrar un modal con un formulario para capturar  un Vale
 */
const vistaNuevoVale = () => {
  const Modal = new PartnersModal(
    {
      title: 'Vales',
      modalSize: 'modal-lg',
      modalId: 'nuevo-vale-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}vista_nuevo_vale`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nuevo_vale_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });

        $('#nuevo_vale_form .input-group.date').datepicker({
          format: 'dd/mm/yyyy',
          todayBtn: 'linked',
          language: 'es',
          daysOfWeekDisabled: '0',
          autoclose: true,
          todayHighlight: true,
          orientation: 'auto',
          container: '.input-group.date',
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para capturar un Vale.
 * @param e
 * @param reporteador
 */
const capturarVale = (e, reporteador) => {
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const datos = $(e.currentTarget).serializeArray();
  const datosEnvio = {};
  for (let i = 0; i < datos.length; i += 1) {
    datosEnvio[datos[i].name] = datos[i].value;
  }

  // Validar fecha
  const fechaUsado = $('#nuevo_vale_form .input-group.date').datepicker('getDate');
  if (!fechaUsado) {
    iziToast.error({
      title: 'Error',
      message: 'Selecciona la fecha.',
    });
    return;
  }

  datosEnvio.fecha_usado = moment(fechaUsado).unix();
  datosEnvio.fecha_usado_end = moment(fechaUsado).endOf('day').unix();
  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_vale`, datosEnvio)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nuevo-vale-modal').modal('hide');
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para generar la tabla de la vista del vales.
 * @param idVehiculo
 * @param start
 * @param end
 */
const reporteVales = (idVehiculo, start, end) => {
  // Mostrar el loader
  const reporteadorVales = $('#reporteador_vales');
  reporteadorVales.html(loader);

  axios.post(`${urls.consultas}reporte_vales`, { idVehiculo, start, end })
    .then((res) => {
      if (res.data.status === 'success') {
        reporteadorVales.html(res.data.message);
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para mostrar un modal con todos los vales de un vehículo.
 * @param e
 */
const vistaVales = (e) => {
  const idVehiculo = $(e.currentTarget).parents('tr').attr('data-id-vehiculo');
  const nombreVehiculo = $(e.currentTarget).parents('tr').attr('data-nombre-vehiculo');
  const start = moment().startOf('month');
  const end = moment().endOf('month');
  const Modal = new PartnersModal(
    {
      title: `Vales ${nombreVehiculo}`,
      modalSize: 'modal-lg',
      modalId: 'vista-vales-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}vista_vales`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);

        const picker = new DateRangePicker('#reportrange_vales', start, end, {}, '.parent-calendar-vales');
        reporteVales(idVehiculo, start, end);
        $(picker.getElement()).on('apply.daterangepicker', (event, datePicker) => {
          picker.updateDates(datePicker.startDate, datePicker.endDate);
          reporteVales(idVehiculo, datePicker.startDate, datePicker.endDate);
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Método para generar una lista de vales usados por un vehiculo sin kilometraje relacionado.
 * @param idVehiculo
 */
const loadValesVehiculo = (idVehiculo) => {
  axios.get(`${urls.consultas}lista_vales_vehiculo?id_vehiculo=${idVehiculo}`)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nuevo_km_form select[name="id_control"]').empty();
        res.data.message.forEach((item) => {
          $('#nuevo_km_form select[name="id_control"]')
            .append(`<option value="${item.id_control}">${item.folio_vale} | ${item.monto_vale}</option>`);
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para generar un modal en el cual se mostrarán todos los vales utilizados
 */
const vistaReporteVales = () => {
  const Modal = new PartnersModal(
    {
      title: 'Reporte Vales',
      modalSize: 'modal-lg',
      modalId: 'vista-vales-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}vista_reporte_vales`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $('#table_reporte_vales').partnersfilter();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

export {
  vistaNuevoVehiculo,
  capturarVehiculo,
  vistaNuevoGasto,
  capturarGasto,
  vistaNuevoSeguro,
  capturarSeguro,
  vistaNuevoKm,
  capturarKm,
  vistaNuevaAfinacion,
  capturarAfinacion,
  vistaNuevoRefrendo,
  capturarRefrendo,
  visualizarArchivo,
  copyToClipboard,
  vistaVehiculo,
  vistaKilometraje,
  vistaGastoAnual,
  vistaNuevaEntradaVales,
  capturarEntradaVales,
  vistaNuevoVale,
  capturarVale,
  vistaVales,
  loadValesVehiculo,
  vistaReporteVales,
};
