/**
 * Clase para generar un reporteador que traerá datos del servidor.
 */
export default class Reporteador {
  /**
     * Crea un reporteador en el cual se usará para traer datos del servidor y mostrarlos en una tabla.
     * @param {string} element El elemento en el cual deberá de ser creado el reporteador.
     * @param {string} url A cual url se deberá de consultar los datos.
     * @param {object} data Objeto con opciones para pasar al servidor
     * @param {instance} axios Instancia de axios la cual se usará para hacer las consultas.
     * @param {function} axiosSuccess Callback que se ejecutará al momento de que un request exitoso en axios.
     * @param {function} axiosError Callback que se ejecutará al momento de que un request sea erróneo.
     */
  constructor(element, url, data, axios, axiosSuccess, axiosError) {
    this.element = element;
    this.url = url;
    this.data = data;
    this.axios = axios;
    this.axiosSuccess = axiosSuccess;
    this.hasBeenRendered = false;
    // Aquí lo recomendable es no mostrar el error al usuario, ya que de eso se encarga
    // el interceptor de la instancia de axios.
    this.axiosError = axiosError;
    this.updating = false;

    this.runReporteador();
  }

  /**
     * Prepará el reporteador ser actualizado, agrega css opacity para simular espera.
     */
  toggleUpdating() {
    if (this.hasBeenRendered) {
      // Solamente agregar el opacity cuando el reporteador si sea una tabla.
      if (document.querySelectorAll(`${this.element} table`)[0]) {
        this.updating = !this.updating;
        if (this.updating === true) {
          document.querySelectorAll(`${this.element} table`)[0].style.opacity = '0.5';
        } else if (this.updating === false) {
          document.querySelectorAll(`${this.element} table`)[0].style.opacity = '1';
        }
      }
    }
  }

  /**
     * Cuando se necesite actualizar los datos del reporteador, se pasa el nuevo objeto de los nuevos datos.
     * @param {object} updatedOptions Objeto con los nuevos datos a ser reemplazados.
     */
  rebuildOptions(updatedOptions) {
    this.data = updatedOptions;
    this.runReporteador();
  }

  /**
     * Método principal donde se ejecuta la consulta del reporteador.
     */
  runReporteador() {
    this.toggleUpdating();
    this.axios.post(this.url, this.data).then((res) => {
      this.toggleUpdating();
      this.axiosSuccess(res);
      document.querySelector(this.element).innerHTML = res.data.message;
      this.hasBeenRendered = true;
    }).catch((err) => {
      this.axiosError(err);
    });
  }

  /**
     * Método usado para actualizar el reporteador sin necesidad de usar rebuildOptions()
     */
  forceUpdate() {
    this.runReporteador();
  }

  /**
     * Regresa el objeto actual de data para posteriormente modificar lo necesario.
     * @returns {object} data
     */
  getCurrentData() {
    return this.data;
  }
}
