import moment from 'moment';
import $ from 'jquery';

require('../../../plugin/js/daterangepicker');

/**
 * Generá una dateRangePicker ya sea con opciones por default, o con personalización personalizada.
 */
export default class dateRangePicker {
  /**
   * Genera un dateRangePicker
   * @param {string} element Elemento en el cual deberá de ser insertado el daterangepicker
   * @param {object} start Objeto de Moment.js usado para indicar la fecha de inicio por default
   * @param {object} end Objeto de Moment.js usado para indicar la fecha final por default
   * @param {object} options Opciones del dateRangePicker
   * @param {string} parentEl  El elemento en el cual deberá quedar todoo agrupado
   */
  constructor(element, start, end, options = {}, parentEl) {
    this.element = element;
    this.start = start;
    this.end = end;
    this.options = {
      autoApply: true,
      linkedCalendars: false,
      parentEl,
      locale: {
        format: 'MM/DD/YYYY',
        separator: ' - ',
        applyLabel: 'Aplicar',
        cancelLabel: 'Cancelar',
        fromLabel: 'Desde',
        toLabel: 'Hasta',
        customRangeLabel: 'Personalizado',
        weekLabel: 'S',
        daysOfWeek: [
          'Do',
          'Lu',
          'Ma',
          'Mi',
          'Ju',
          'Vi',
          'Sa',
        ],
        monthNames: [
          'Enero',
          'Febrero',
          'Marzo',
          'Abril',
          'Mayo',
          'Junio',
          'Julio',
          'Agosto',
          'Septiembre',
          'Octubre',
          'Noviembre',
          'Diciembre',
        ],
        firstDay: 1,
      },
      startDate: this.start,
      endDate: this.end,
      showDropdowns: true,
      ranges: {
        Enero: [moment().month('January').startOf('month'), moment().month('January').endOf('month')],
        Julio: [moment().month('July').startOf('month'), moment().month('July').endOf('month')],
        Hoy: [moment(), moment()],
        Febrero: [moment().month('February').startOf('month'), moment().month('February').endOf('month')],
        Agosto: [moment().month('August').startOf('month'), moment().month('August').endOf('month')],
        'Esta Semana': [moment().startOf('week'), moment().endOf('week')],
        Marzo: [moment().month('March').startOf('month'), moment().month('March').endOf('month')],
        Septiembre: [moment().month('September').startOf('month'), moment().month('September').endOf('month')],
        'Este Mes': [moment().startOf('month'), moment().endOf('month')],
        Abril: [moment().month('April').startOf('month'), moment().month('April').endOf('month')],
        Octubre: [moment().month('October').startOf('month'), moment().month('October').endOf('month')],
        'Este Bimestre': [moment().startOf('month'), moment().add(1, 'month').endOf('month')],
        Mayo: [moment().month('May').startOf('month'), moment().month('May').endOf('month')],
        Noviembre: [moment().month('November').startOf('month'), moment().month('November').endOf('month')],
        'Este Año': [moment().startOf('year'), moment().endOf('year')],
        Junio: [moment().month('June').startOf('month'), moment().month('June').endOf('month')],
        Diciembre: [moment().month('December').startOf('month'), moment().month('December').endOf('month')],
      },
    };
    if (typeof options === 'object' && !$.isEmptyObject(options)) {
      this.options = options;
    }
    this.initDateRangePicker();
  }

  /**
     * Inicializa el dateRangePicker en el elemento que se indicó.
     */
  initDateRangePicker() {
    // Para inicializar los valores en el input
    this.callback();

    // noinspection JSUnresolvedFunction
    $(this.element).daterangepicker(this.options, this.callback());
  }

  /**
     * Regresa el elemento que se pasó al momento de crear la instancia de este dateRangePicker
     * @returns {string|*}
     */
  getElement() {
    return this.element;
  }

  /**
   * Actualiza las fechas por las seleccionadas, se usa en el evento de apply.daterangepicker,
   * una vez se da click en el evento, hay que actualizar las fechas y lanzar el callback para
   * actualizar el input con las nuevas fechas.
   * @param start
   * @param end
   */
  updateDates(start, end) {
    this.start = start;
    this.end = end;
    this.callback();
  }

  /**
     * Callback a ser ejecutada cuando se necesite, ya sea cuando se cree el dateRangePicker o en eventos
     */
  callback() {
    $(`${this.element} span`).html(`${this.start.format('DD/MM/YYYY')} - ${this.end.format('DD/MM/YYYY')}`);
  }
}
