import $ from 'jquery';
import iziToast from 'izitoast';
import Cleave from 'cleave.js';
import moment from 'moment';
import numbro from 'numbro';
import Swal from 'sweetalert2';
import PartnersModal from '../utilities/js/partnersModal';
import '../utilities/js/partnersBtn';
import '../../plugin/js/jquery.form-validator';
import { axiosInstance as axios, urls } from './config/config';
import 'jquery-typeahead';
import 'jquery-typeahead/dist/jquery.typeahead.min.css';
import partnersBuscador from '../utilities/js/partnersBuscador';


/**
 * Array con una lista de objetos que contienen los equipos que se van a respaldar.
 * @type {Array}
 */
let listEquipos = [];


/**
 * Genera una vista para capturar un nuevo servicio de respaldo.
 */
const vistaNuevoBackup = () => {
  const Modal = new PartnersModal(
    {
      title: 'Nuevo Servicio de Respaldo',
      modalSize: 'modal-lg',
      modalId: 'nuevo-backup-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();

  axios.get(`${urls.consultas}nuevo_backup`)
    .then((res) => {
      if (res.data.status === 'success') {
        Modal.updateBody(res.data.message);
        $.validate({
          form: '#nuevo_backup_form',
          errorMessageClass: 'invalid-feedback',
          inputParentClassOnError: 'text-danger',
          inputParentClassOnSuccess: 'text-success',
          errorElementClass: 'is-invalid',
          successElementClass: 'is-valid',
        });

        partnersBuscador('#buscar_cliente', '/system/tools/lista_clientes/complete');
        listEquipos = [];
        // eslint-disable-next-line no-new
        new Cleave('#nuevo_backup_form input[name="precio_renovacion"]', {
          numeral: true,
          numeralThousandsGroupStyle: 'thousand',
        });
        $('#nuevo_backup_form .input-group.date').datepicker({
          format: 'dd/mm/yyyy',
          todayBtn: 'linked',
          language: 'es',
          daysOfWeekDisabled: '0',
          autoclose: true,
          todayHighlight: true,
          orientation: 'auto',
          container: '.input-group.date',
        });
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para calcular todas las fechas en las que se deberá crear
 * una renovación.
 * @param intervalo
 * @param frecuenciaTipo
 * @param fechaInicio
 * @returns {Array}
 */
const getArrayFechas = (intervalo, frecuenciaTipo, fechaInicio) => {
  let count;

  if (frecuenciaTipo === 'WEEKLY') {
    count = Math.round(260 / intervalo);
  } else if (frecuenciaTipo === 'MONTHLY') {
    count = Math.round(120 / intervalo);
  } else if (frecuenciaTipo === 'YEARLY') {
    count = Math.round(20 / intervalo);
  }
  // eslint-disable-next-line no-param-reassign
  fechaInicio = moment(fechaInicio);
  let dia = fechaInicio.date();
  if (dia <= 9) {
    dia = `0${fechaInicio.date()}`;
  }
  let mes = (fechaInicio.month() + 1);
  if (mes <= 9) {
    mes = `0${fechaInicio.month() + 1}`;
  }

  const year = fechaInicio.year();

  let rules;
  if (frecuenciaTipo === 'MONTHLY' && (+dia === 29 || +dia === 30 || +dia === 31)) {
    rules = new window.RRule(window.RRule.parseString(`FREQ=${frecuenciaTipo};DTSTART=${year}${mes}${dia}T214000Z;COUNT=${count};INTERVAL=${intervalo};BYMONTHDAY=-1`));
    iziToast.warning({
      title: 'Recuerda',
      message: `Algunos meses tienen menos de ${dia} días.Por lo tanto, la repetición tendrá lugar en el último día de cada mes.`,
    });
  } else {
    rules = new window.RRule(window.RRule.parseString(`FREQ=${frecuenciaTipo};DTSTART=${year}${mes}${dia}T060000Z;COUNT=${count};INTERVAL=${intervalo}`));
  }
  const fechasArray = rules.all();
  const fechasLength = fechasArray.length;
  const newDateArray = [];
  for (let i = 0; i < fechasLength; i++) {
    newDateArray[i] = new Date(fechasArray[i]).getTime() / 1000;
  }

  return newDateArray;
};

/**
 * Función para capturar un nuevo respaldo
 * @param e
 * @param reporteador
 */
const capturarBackup = (e, reporteador) => {
  e.preventDefault();
  const btn = $(e.currentTarget).find('button[type="submit"]');
  const datos = $(e.currentTarget).serializeArray();
  const datosEnvio = {};
  for (let i = 0; i < datos.length; i += 1) {
    datosEnvio[datos[i].name] = datos[i].value;
  }

  // Validar Cliente y Contacto
  if (!$('#nuevo_backup_form #buscar_cliente').attr('data-id-cliente')) {
    iziToast.error({
      title: 'Error',
      message: 'Selecciona un cliente',
    });
    return;
  }
  datosEnvio.id_cliente = $('#nuevo_backup_form #buscar_cliente').attr('data-id-cliente');

  const tipoNegocio = +($('#nuevo_backup_form #buscar_cliente').attr('data-tipo-negocio'));
  if (tipoNegocio === 1 || tipoNegocio === 3) {
    const idContacto = +($('.select_lista_contactos').find('option:selected').val());
    if (idContacto === 0 || !idContacto) {
      iziToast.error({
        title: 'Error',
        message: 'Selecciona un contacto',
      });
      return;
    }
    datosEnvio.id_contacto = idContacto;
  }

  // Validar Fecha Renovación
  const fechaRenovacion = $('#nuevo_backup_form .input-group.date').datepicker('getDate');
  if (!fechaRenovacion) {
    iziToast.error({
      title: 'Error',
      message: 'Selecciona la fecha renovación.',
    });
    return;
  }
  datosEnvio.fecha_renovacion = moment(fechaRenovacion).unix();

  // Validar que la cantidad total de espacio distribuido entre los equipos seleccionados
  // sea exactamente igual a la cantidad del paquete
  let totalPaquete;
  if (+datosEnvio.paquete_respaldo === 1) {
    totalPaquete = 300;
  } else if (+datosEnvio.paquete_respaldo === 2) {
    totalPaquete = 500;
  } else if (+datosEnvio.paquete_respaldo === 3) {
    totalPaquete = 1000;
  } else if (+datosEnvio.paquete_respaldo === 4) {
    totalPaquete = 2000;
  }

  let totalUsoEspacio = 0;
  listEquipos.forEach((item) => {
    totalUsoEspacio += +item.uso_espacio;
  });

  if (totalPaquete !== totalUsoEspacio) {
    iziToast.error({
      title: 'Error',
      message: `Debes de llenar exactamente todos los GB del paquete que seleccionaste (${totalUsoEspacio} / ${totalPaquete})`,
    });
    return;
  }
  datosEnvio.totalUsoEspacio = totalUsoEspacio;
  datosEnvio.listEquipos = listEquipos;
  datosEnvio.precio_renovacion = numbro(datosEnvio.precio_renovacion).value();
  datosEnvio.dates = getArrayFechas(datosEnvio.input_frecuencia, datosEnvio.select_frecuencia, fechaRenovacion);

  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}capturar_respaldo`, datosEnvio)
    .then((res) => {
      if (res.data.status === 'success') {
        $('#nuevo-backup-modal').modal('hide');
        setTimeout(() => {
          // eslint-disable-next-line no-use-before-define
          vistaFichaInformativa(res.data.message);
        }, 400);
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Función para calcular y mostrar el espacio disponible
 */
const renderUsoEspacio = () => {
  let totalUsoEspacio = 0;
  const paquete = +($('select[name="paquete_respaldo"] option:selected').val());
  listEquipos.forEach((item) => {
    totalUsoEspacio += +item.uso_espacio;
  });
  let paqueteSize;
  if (paquete === 1) {
    paqueteSize = 300;
  } else if (paquete === 2) {
    paqueteSize = 500;
  } else if (paquete === 3) {
    paqueteSize = 1000;
  } else if (paquete === 4) {
    paqueteSize = 2000;
  }

  // Mostrar total Uso disco
  const color = totalUsoEspacio > paqueteSize ? 'text-danger' : '';
  $('.container_uso_disco').html(`<p class="lead text-center ${color}">${totalUsoEspacio} / ${paqueteSize} GB Usados</p>`);
};

/**
 * Función para generar la lista de equipos que se tiene cargada.
 */
const renderListaEquipos = () => {
  const container = $('#container_list_equipos');
  let containerTpl = '';
  listEquipos.forEach((item) => {
    containerTpl += `<div class="row mb-1">
                <div class="col-8">
                    <input type="text" class="form-control equipo_nombre" value="${item.nombre_equipo}" data-id-equipo="${item.id}"/>
                </div>
                <div class="col-2">
                  <input type="number" min="1" class="form-control equipo_uso" value="${item.uso_espacio}" data-id-equipo="${item.id}"/>
                </div>
              </div>`;
  });
  container.html(containerTpl);
  renderUsoEspacio();
};

/**
 * Función para controlar los equipos que se capturarán en el respaldo.
 * @param e
 */
const handleNumeroEquipos = (e) => {
  const numberEquipos = +($(e.currentTarget).val());
  for (let i = 0; i < numberEquipos; i++) {
    if (numberEquipos > listEquipos.length) {
      listEquipos.push({
        id: listEquipos.length + 1,
        nombre_equipo: `Equipo ${listEquipos.length + 1}`,
        uso_espacio: 0,
      });
    } else if (numberEquipos < listEquipos.length) {
      listEquipos.pop();
    }
  }

  // Mostrar Form Renovación
  $('.container_renovacion').show();
  renderListaEquipos();
};

/**
 * Función que se encarga de controlar el uso de los equipos en la lista
 * de equipos.
 * @param e
 */
const handleEquipoUsoChange = (e) => {
  const val = $(e.currentTarget).val();
  const id = $(e.currentTarget).attr('data-id-equipo');

  listEquipos = listEquipos.map((item) => {
    if (+item.id === +id) {
      // eslint-disable-next-line no-param-reassign
      item.uso_espacio = val;
    }
    return item;
  });
  renderUsoEspacio();
};

/**
 * Función que se encarga de controlar los nombres de los equipos de la lista
 * de equipos.
 * @param e
 */
const handleEquipoNombreChange = (e) => {
  const val = $(e.currentTarget).val();
  const id = $(e.currentTarget).attr('data-id-equipo');

  listEquipos = listEquipos.map((item) => {
    if (+item.id === +id) {
      // eslint-disable-next-line no-param-reassign
      item.nombre_equipo = val;
    }
    return item;
  });
};

/**
 * Función para mostrar el container de los equipos una vez que se seleccione un paquete.
 */
const showEquiposContainer = () => {
  $('.equipos_container').show();
  renderUsoEspacio();
};

const generateVistaEquipos = (idBackup, Modal = false) => {
  axios.get(`${urls.consultas}vista_equipos?id_backup=${idBackup}`)
    .then((res) => {
      if (res.data.status === 'success') {
        if (!Modal) {
          $('#vista-equipos-modal .modal-body').html(res.data.message);
        } else {
          Modal.updateBody(res.data.message);
        }
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para mostrar un modal con información sobre los equipos que se están respaldando.
 * @param idBackup
 */
const vistaEquipos = (idBackup) => {
  const Modal = new PartnersModal(
    {
      title: 'Equipos de Servicio de Respaldo',
      modalSize: 'modal-lg',
      modalId: 'vista-equipos-modal',
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();
  generateVistaEquipos(idBackup, Modal);
};

/**
 * Función para consultar el uso real que tiene un equipo en AWS.
 * @param idBackup
 * @param idEquipo
 * @param e
 */
const consultarUsoEquipo = (idBackup, idEquipo, e) => {
  $(e.currentTarget)
    .html('<div class="spinner-border text-primary"></div>');

  axios.get(`${urls.consultas}consultar_espacio_equipo?id_equipo=${idEquipo}`)
    .then((res) => {
      if (res.data.status === 'success') {
        generateVistaEquipos(idBackup);
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
        generateVistaEquipos(idBackup);
      }
    });
};

/**
 * Función para consultar el espacio usado de un bucket.
 * @param idBackup
 * @param e
 * @param reporteador
 */
const consultarUsoBucket = (idBackup, e, reporteador) => {
  $(e.currentTarget)
    .html('<div class="spinner-border text-primary"></div>');

  axios.get(`${urls.consultas}consultar_espacio_bucket?id_backup=${idBackup}`)
    .then((res) => {
      if (res.data.status === 'success') {
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      }
    });
};

/**
 * Lista para guardar el estado de todos los equipos que se están editando actualmente.
 * @type {Array}
 */
let listaEquiposEditando = [];

/**
 * Función para mostrar el total del espacio asignado y distribuido entre los equipos.
 * @param totalEspacio El total del espacio que se puede distribuir entre los equipos.
 */
const handleTotalAsignadoEditar = (totalEspacio) => {
  let totalAsingado = 0;
  listaEquiposEditando.forEach((item) => {
    if (!item.deleted) totalAsingado += item.espacioEquipo;
  });
  let totalPaquete;
  if (+totalEspacio === 1) {
    totalPaquete = 300;
  } else if (+totalEspacio === 2) {
    totalPaquete = 500;
  } else if (+totalEspacio === 3) {
    totalPaquete = 1000;
  } else if (+totalEspacio === 4) {
    totalPaquete = 2000;
  }
  if (totalAsingado > totalPaquete) {
    $('.total_espacio_edit').html(`<p class="lead text-danger">${totalAsingado} / ${totalPaquete} GB</p>`);
  } else {
    $('.total_espacio_edit').html(`<p class="lead">${totalAsingado} / ${totalPaquete} GB</p>`);
  }
};

/**
 * Función que se encarga de actualizar la lista de equipos con el nuevo nombre.
 * @param e
 */
const handleEditNombreEquipo = (e) => {
  const tr = $(e.currentTarget);
  const nuevoNombre = tr.val();
  const idEquipo = tr.closest('tr').attr('data-id-equipo');
  listaEquiposEditando = listaEquiposEditando.map((item) => {
    if (+item.idEquipo === +idEquipo) {
      // eslint-disable-next-line no-param-reassign
      item.nombreEquipo = nuevoNombre;
    }
    return item;
  });
};

/**
 * Función que se encarga de actualizar la lista de equipos con el nuevo espacio.
 * @param e
 */
const handleEditEspacioEquipo = (e) => {
  const tr = $(e.currentTarget);
  const nuevoEspacio = tr.val();
  const idEquipo = tr.closest('tr').attr('data-id-equipo');
  const table = $(e.currentTarget).closest('.vista_equipos_container').find('.table_equipos_editing');
  const paquete = table.attr('data-paquete');
  listaEquiposEditando = listaEquiposEditando.map((item) => {
    if (+item.idEquipo === +idEquipo) {
      // eslint-disable-next-line no-param-reassign
      item.espacioEquipo = +nuevoEspacio;
    }
    return item;
  });

  // Mostrar total GB asignados
  handleTotalAsignadoEditar(paquete);
};

/**
 * Función para eliminar un euqipo al momento de editar.
 * @param e
 */
const eliminarEquipoEditar = (e) => {
  const tr = $(e.currentTarget);
  const idEquipo = tr.closest('tr').attr('data-id-equipo');
  const table = $(e.currentTarget).closest('.vista_equipos_container').find('.table_equipos_editing');
  const paquete = table.attr('data-paquete');
  listaEquiposEditando = listaEquiposEditando.map((item) => {
    if (+item.idEquipo === +idEquipo) {
      // eslint-disable-next-line no-param-reassign
      item.deleted = true;
    }
    return item;
  }).filter(item => !(item.newEquipo && item.deleted));


  // Eliminar el tr de la tabla
  table.find('tbody tr').toArray().forEach((currentTr) => {
    if (+$(currentTr).attr('data-id-equipo') === +idEquipo) {
      $(currentTr).remove();
    }
  });

  // Mostrar total GB asignados
  handleTotalAsignadoEditar(paquete);
};

/**
 * Función para cancelar la edición de los equipos de un servicio de respaldo.
 * @param e
 */
const cancelarEdicion = (e) => {
  const idBackup = $(e.currentTarget).attr('data-id-backup');
  listaEquiposEditando = [];
  vistaEquipos(idBackup);
};

/**
 * Función para agregar formato a los input de numeros y prevenir que pongan
 * algo que no sea numeros.
 */
const formatNumberInput = () => {
  // Agregar formato a los campos de numeros
  $('.espacio_equipo_editing').toArray().forEach((item) => {
    // eslint-disable-next-line no-new
    new Cleave(item, {
      numeral: true,
      numeralDecimalMark: '',
      delimiter: '',
    });
  });
};

/**
 * Función para generar una vista para editar los equipos de un servicio de respaldo.
 * @param e
 */
const vistaEditarEquipos = (e) => {
  listaEquiposEditando = [];
  const table = $(e.currentTarget).closest('.vista_equipos_container').find('.table_equipos');
  const paquete = table.attr('data-paquete');
  table.find('tbody tr').toArray()
    .forEach((v) => {
      const a = $(v);
      listaEquiposEditando.push({
        idEquipo: +a.attr('data-id-equipo'),
        nombreEquipo: a.attr('data-nombre'),
        espacioEquipo: +a.attr('data-espacio'),
        usoEquipo: +a.attr('data-uso-real'),
        deleted: false,
        newEquipo: false,
      });
    });

  $('.table_equipos').hide();
  $(e.currentTarget).hide();
  $('.cancelar_edicion_equipos_respaldo').show();
  $('.guardar_edicion_equipos_respaldo').show();

  // Mostrar total GB asignados
  handleTotalAsignadoEditar(paquete);

  // Mostrar la tabla
  $('.table_equipos_editing').show();
  $('.table_equipos_editing tbody').empty();
  listaEquiposEditando.forEach((item) => {
    $('.table_equipos_editing tbody')
      .append(`
        <tr data-id-equipo="${item.idEquipo}">
            <td><input class="form-control nombre_equipo_editing" value="${item.nombreEquipo}" /></td>
            <td><input class="form-control espacio_equipo_editing text-right" value="${item.espacioEquipo}"/></td>
            <td><input class="form-control text-right" readonly value="${item.usoEquipo}" /></td>
            <td class="text-center cursor-pointer eliminar_equipo_edicion"><i class="fas fa-trash-alt"></i></td>
        </tr>
      `);
  });

  // Agregar hasta al ultimo un tr que servirá para agregar más equipos.
  $('.table_equipos_editing tbody')
    .append(`
        <tr>
            <td colspan="4"><button class="btn btn-success btn-sm btn-left agregar_nuevo_equipo_edicion">
            <i class="fas fa-plus"></i> Equipo</button></td>
        </tr>
      `);

  formatNumberInput();
};

/**
 * Función para agregar un nuevo equipo al servicio de respaldo.
 */
const agregarEquipoEdicion = () => {
  const id = new Date().getTime();
  $(`
        <tr data-id-equipo="${id}">
            <td><input class="form-control nombre_equipo_editing" value="" /></td>
            <td><input class="form-control espacio_equipo_editing text-right" value=""/></td>
            <td><input class="form-control text-right" readonly value="" /></td>
            <td class="text-center cursor-pointer eliminar_equipo_edicion"><i class="fas fa-trash-alt"></i></td>
        </tr>
      `)
    .insertBefore($('.table_equipos_editing').find('.agregar_nuevo_equipo_edicion').closest('tr'));

  // Agregar el nuevo tr a la lista
  listaEquiposEditando.push({
    idEquipo: id,
    nombreEquipo: '',
    espacioEquipo: 0,
    usoEquipo: 0,
    deleted: false,
    newEquipo: true,
  });

  formatNumberInput();
};

/**
 * Función para capturar los cambios realizados a los equipos de un servicio de respaldo.
 * @param e
 * @param reporteador
 */
const guardarEdicionEquipos = (e, reporteador) => {
  const btn = $(e.currentTarget);
  const idBackup = btn.attr('data-id-backup');
  let tDel = 0;

  // Validar que por lo menos un equipo exista.
  listaEquiposEditando.forEach((item) => {
    if (item.deleted) tDel++;
  });
  if (tDel === listaEquiposEditando.length) {
    iziToast.error({
      title: 'Error',
      message: 'Ingresa por lo menos un equipo!',
    });
    return;
  }

  // Validar que se este usando tod0 el espacio
  const table = $(e.currentTarget).closest('.vista_equipos_container').find('.table_equipos_editing');
  const totalEspacio = table.attr('data-paquete');
  let totalPaquete;
  if (+totalEspacio === 1) {
    totalPaquete = 300;
  } else if (+totalEspacio === 2) {
    totalPaquete = 500;
  } else if (+totalEspacio === 3) {
    totalPaquete = 1000;
  } else if (+totalEspacio === 4) {
    totalPaquete = 2000;
  }

  let totalAsingado = 0;
  listaEquiposEditando.forEach((item) => {
    if (!item.deleted) totalAsingado += item.espacioEquipo;
  });

  if (+totalAsingado !== totalPaquete) {
    iziToast.error({
      title: 'Error',
      message: 'Tienes que distribuir la cantidad exacta de GB entre todos los equipos!',
    });
    return;
  }


  // Validar que todos los equipos no eliminados de la lista tengan datos (id, nombre y espacio)
  listaEquiposEditando.forEach((item) => {
    if (item.deleted) totalAsingado += item.espacioEquipo;
  });

  let err = false;
  for (let i = 0; i < listaEquiposEditando.length; i++) {
    if (listaEquiposEditando[i].deleted === false) {
      if ($.trim(listaEquiposEditando[i].nombreEquipo) === '' || listaEquiposEditando[i].espacioEquipo <= 0) {
        err = true;
        break;
      }
    }
  }

  if (err) {
    iziToast.error({
      title: 'Error',
      message: 'Ingresa el nombre del equipo y el espacio de almacenamiento a TODOS los equipos que aparecen!',
    });
    return;
  }

  const datosEnvio = {};
  datosEnvio.listEquipos = listaEquiposEditando;
  datosEnvio.idBackup = idBackup;
  btn.partnersBtnControl('start', btn);

  // ya estan todas las validaciones necesarias, ya solo falta capturar los datos
  axios.post(`${urls.consultas}capturar_edicion_equipos`, datosEnvio)
    .then((res) => {
      if (res.data.status === 'success') {
        vistaEquipos(idBackup);
        iziToast.success({
          title: 'Bien!',
          message: res.data.message,
        });
        reporteador.forceUpdate();
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

/**
 * Método para cambiar un paquete de un backup.
 * @param e
 * @param reporteador
 */
const cambiarPaquete = (e, reporteador) => {
  const tr = $(e.currentTarget).parents('tr');
  const nombreBucket = tr.attr('data-nombre-bucket');
  const idBackup = tr.attr('data-id-backup');

  Swal.fire({
    title: `Modificar paquete de ${nombreBucket}`,
    input: 'select',
    inputOptions: {
      1: '300 GB',
      2: '500 GB',
      3: '1 TB',
      4: '2 TB',
    },
    inputPlaceholder: 'Selecciona un paquete',
    showCancelButton: true,
    cancelButtonText: 'Cancelar',
    confirmButtonText: 'Guardar',
    showLoaderOnConfirm: true,
    allowOutsideClick: () => !Swal.isLoading(),
    inputValidator: value => new Promise((resolve) => {
      if (value) {
        resolve();
      } else {
        resolve('Selecciona un paquete!');
      }
    }),
    preConfirm: paquete => axios.post(`${urls.consultas}capturar_edicion_paquete`, { idBackup, paquete })
      .then((res) => {
        if (res.data.status === 'success') {
          reporteador.forceUpdate();
          return res.data;
        } if (res.data.status === 'error') {
          throw new Error(res.data.message);
        } else {
          throw new Error('Ocurrió un error, inténtalo otra vez :(');
        }
      }).catch((error) => {
        Swal.showValidationMessage(
          `Request failed: ${error}`,
        );
      }),
  }).then((result) => {
    if (result) {
      Swal.fire({
        type: 'success',
        title: result.value.message,
        timer: 1000,
      });
    }
  });
};

/**
 * Función para marcar un registro para que sea eliminado después.
 * @param e
 * @param reporteador
 */
const preEliminarServicio = (e, reporteador) => {
  const tr = $(e.currentTarget).parents('tr');
  const nombreBucket = tr.attr('data-nombre-bucket');
  const idBackup = tr.attr('data-id-backup');

  Swal.fire({
    title: `Deseas marcar a ${nombreBucket} para una futura eliminación?`,
    type: 'warning',
    showCancelButton: true,
    cancelButtonText: 'Cancelar',
    confirmButtonText: 'Eliminar',
    showLoaderOnConfirm: true,
    allowOutsideClick: () => !Swal.isLoading(),
    preConfirm: () => axios.post(`${urls.consultas}pre_eliminar_backup`, { idBackup })
      .then((res) => {
        if (res.data.status === 'success') {
          reporteador.forceUpdate();
          return res.data;
        } if (res.data.status === 'error') {
          throw new Error(res.data.message);
        } else {
          throw new Error('Ocurrió un error, inténtalo otra vez :(');
        }
      }).catch((error) => {
        Swal.showValidationMessage(
          `Request failed: ${error}`,
        );
      }),
  }).then((result) => {
    if (result) {
      Swal.fire({
        type: 'success',
        title: result.value.message,
        timer: 1000,
      });
    }
  });
};

/**
 * Función para eliminar permanentemente un registro.
 * @param e
 * @param reporteador
 */
const eliminarServicio = (e, reporteador) => {
  const tr = $(e.currentTarget).parents('tr');
  const nombreBucket = tr.attr('data-nombre-bucket');
  const idBackup = tr.attr('data-id-backup');

  Swal.fire({
    title: `Deseas eliminar permanentemente el respaldo de ${nombreBucket} ?`,
    type: 'warning',
    showCancelButton: true,
    cancelButtonText: 'Cancelar',
    confirmButtonText: 'Eliminar Permanentemente',
    showLoaderOnConfirm: true,
    allowOutsideClick: () => !Swal.isLoading(),
    preConfirm: () => axios.post(`${urls.consultas}eliminar_backup`, { idBackup })
      .then((res) => {
        if (res.data.status === 'success') {
          reporteador.forceUpdate();
          return res.data;
        } if (res.data.status === 'error') {
          throw new Error(res.data.message);
        } else {
          throw new Error('Ocurrió un error, inténtalo otra vez :(');
        }
      }).catch((error) => {
        Swal.showValidationMessage(
          `Request failed: ${error}`,
        );
      }),
  }).then((result) => {
    if (result) {
      Swal.fire({
        type: 'success',
        title: result.value.message,
        timer: 1000,
      });
    }
  });
};

/**
 * Función para activar una llave de acceso en AWS.
 * @param e
 * @param reporteador
 */
const activarAccessKey = (e, reporteador) => {
  const tr = $(e.currentTarget).parents('tr');
  const nombreBucket = tr.attr('data-nombre-bucket');
  const idBackup = tr.attr('data-id-backup');

  Swal.fire({
    title: `Deseas activar la llave de ${nombreBucket} ?`,
    type: 'question',
    showCancelButton: true,
    cancelButtonText: 'Cancelar',
    confirmButtonText: 'Si',
    showLoaderOnConfirm: true,
    allowOutsideClick: () => !Swal.isLoading(),
    preConfirm: () => axios.post(`${urls.consultas}cambiar_status_access_key`, { idBackup, status: 1 })
      .then((res) => {
        if (res.data.status === 'success') {
          reporteador.forceUpdate();
          return res.data;
        } if (res.data.status === 'error') {
          throw new Error(res.data.message);
        } else {
          throw new Error('Ocurrió un error, inténtalo otra vez :(');
        }
      }).catch((error) => {
        Swal.showValidationMessage(
          `Request failed: ${error}`,
        );
      }),
  }).then((result) => {
    if (result) {
      Swal.fire({
        type: 'success',
        title: result.value.message,
        timer: 1000,
      });
    }
  });
};

/**
 * Función para inactivar una llave de acceso en AWS.
 * @param e
 * @param reporteador
 */
const inactivarAccessKey = (e, reporteador) => {
  const tr = $(e.currentTarget).parents('tr');
  const nombreBucket = tr.attr('data-nombre-bucket');
  const idBackup = tr.attr('data-id-backup');

  Swal.fire({
    title: `Deseas deshabilitar la llave de ${nombreBucket} ?`,
    type: 'warning',
    showCancelButton: true,
    cancelButtonText: 'Cancelar',
    confirmButtonText: 'Si',
    showLoaderOnConfirm: true,
    allowOutsideClick: () => !Swal.isLoading(),
    preConfirm: () => axios.post(`${urls.consultas}cambiar_status_access_key`, { idBackup, status: 0 })
      .then((res) => {
        if (res.data.status === 'success') {
          reporteador.forceUpdate();
          return res.data;
        } if (res.data.status === 'error') {
          throw new Error(res.data.message);
        } else {
          throw new Error('Ocurrió un error, inténtalo otra vez :(');
        }
      }).catch((error) => {
        Swal.showValidationMessage(
          `Request failed: ${error}`,
        );
      }),
  }).then((result) => {
    if (result) {
      Swal.fire({
        type: 'success',
        title: result.value.message,
        timer: 1000,
      });
    }
  });
};

/**
 * Función para generar la vista de la ficha de un backup.
 * @param idBackup
 * @param el
 */
const renderFichaInformativa = (idBackup, el) => {
  axios.get(`${urls.consultas}vista_ficha_backup?id_backup=${idBackup}`)
    .then((res) => {
      if (res.data.status === 'success') {
        $(el).html(res.data.message);
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      }
    });
};

/**
 * Función para abrir un modal en el cual se mostrará la ficha de un backup.
 * @param idBackup
 */
const vistaFichaInformativa = (idBackup) => {
  const Modal = new PartnersModal(
    {
      title: 'Información Respaldo',
      modalSize: 'modal-lg',
      modalId: 'vista-informacion-respaldo-modal',
      staticBackdrop: true,
      onModalMounted(modalId, callback) {
        // noinspection JSUnresolvedFunction
        $(`#${modalId}`).modal('show').on('hidden.bs.modal', () => callback(modalId));
      },
    },
  );

  Modal.openModal();
  renderFichaInformativa(idBackup, '#vista-informacion-respaldo-modal .modal-body');
};

/**
 * Función para copiar un texto al clipboard.
 * @param el El elemento que tiene el texto que se va a copiar
 * @param container El container en donde se pondrá temporalmente el textarea necesario para copiar al clipboard.
 */
const copyToKeyBoard = (el, container = false) => {
  const $this = $(el);
  const $temp = $(`<textarea>${$this.text()}</textarea>`);
  if (container) {
    $(container).append($temp);
  } else {
    $('body').append($temp);
  }
  $temp.select();
  document.execCommand('copy');
  $temp.remove();
  $this.tooltip({
    trigger: 'click',
    placement: 'top',
  });

  $this.tooltip('hide')
    .attr('data-original-title', 'Copiado!')
    .tooltip('show');
  setTimeout(() => {
    $this.tooltip('dispose');
  }, 1000);
};

/**
 * Función para enviar por telegram un token a un técnico para que pueda consultar la ficha de un
 * servicio de respaldo.
 * @param e
 * @param reporteador
 * @returns {Promise<void>}
 */
const sendSecretKey = async (e, reporteador) => {
  const tr = $(e.currentTarget).parents('tr');
  const nombreBucket = tr.attr('data-nombre-bucket');
  const idBackup = tr.attr('data-id-backup');
  $(e.currentTarget).html('<div class="spinner-border text-primary"></div>');
  const data = await axios.get(`${urls.consultas}lista_tecnicos`)
    .then(res => res.data.message);
  $(e.currentTarget).html('<i class="fas fa-key"></i>');
  Swal.fire({
    title: `Enviar Token de acceso para ${nombreBucket} al técnico:`,
    input: 'select',
    inputOptions: data,
    inputPlaceholder: 'Selecciona un técnico',
    showCancelButton: true,
    cancelButtonText: 'Cancelar',
    confirmButtonText: 'Guardar',
    showLoaderOnConfirm: true,
    allowOutsideClick: () => !Swal.isLoading(),
    inputValidator: value => new Promise((resolve) => {
      if (value) {
        resolve();
      } else {
        resolve('Selecciona un técnico!');
      }
    }),
    preConfirm: uid => axios.post(`${urls.consultas}send_token_secret_key`, { idBackup, uid })
      .then((res) => {
        if (res.data.status === 'success') {
          reporteador.forceUpdate();
          return res.data;
        } if (res.data.status === 'error') {
          throw new Error(res.data.message);
        } else {
          throw new Error('Ocurrió un error, inténtalo otra vez :(');
        }
      }).catch((error) => {
        Swal.showValidationMessage(
          `Request failed: ${error}`,
        );
      }),
  }).then((result) => {
    if (result) {
      Swal.fire({
        type: 'success',
        title: result.value.message,
        timer: 1000,
      });
    }
  });
};

const validarCodigo = (e, codigo) => {
  const btn = $(e.currentTarget);

  btn.partnersBtnControl('start', btn);

  axios.post(`${urls.consultas}validar_codigo_ficha`, { codigo })
    .then((res) => {
      if (res.data.status === 'success') {
        renderFichaInformativa(res.data.message, '.main_container_ficha');
      } else if (res.data.status === 'error') {
        iziToast.error({
          title: 'Error',
          message: res.data.message,
        });
      } else {
        iziToast.error({
          title: 'Error',
          message: 'Ocurrió un error, inténtalo otra vez :(',
        });
      }
      btn.partnersBtnControl('end', btn);
    });
};

export {
  vistaNuevoBackup,
  capturarBackup,
  handleNumeroEquipos,
  handleEquipoUsoChange,
  handleEquipoNombreChange,
  showEquiposContainer,
  vistaEquipos,
  consultarUsoEquipo,
  consultarUsoBucket,
  vistaEditarEquipos,
  handleEditNombreEquipo,
  handleEditEspacioEquipo,
  eliminarEquipoEditar,
  cancelarEdicion,
  guardarEdicionEquipos,
  agregarEquipoEdicion,
  cambiarPaquete,
  preEliminarServicio,
  eliminarServicio,
  activarAccessKey,
  inactivarAccessKey,
  vistaFichaInformativa,
  copyToKeyBoard,
  sendSecretKey,
  validarCodigo,
};
