// Follow the UMD template https://github.com/umdjs/umd/blob/master/templates/returnExportsGlobal.js
(function (root, factory) {
  if (typeof define === 'function' && define.amd) {
    // AMD. Register as an anonymous module.
    define(['jquery'], function (b) {
      return (root.partnersCalc = factory(jquery));
    });
  } else if (typeof module === 'object' && module.exports) {
    // Node. Does not work with strict CommonJS, but
    // only CommonJS-like environments that support module.exports,
    // like Node.

    // Node / Browserify
    //isomorphic issue
    let jQuery = (typeof window != 'undefined') ? window.jQuery : undefined;
    if (!jQuery) {
      jQuery = require('jquery');
      if (!jQuery.fn) jQuery.fn = {};
    }
    module.exports = factory(jQuery);
  } else {
    // Browser globals
    root.partnersCalc = factory(root.jQuery);
  }
}(this, function ($) {
  let PartnersCalc = function (el, options) {
    this.el = el;
    this.isActive = false;
    this.displayActiveValue = 0;
    this.listaOperaciones = [];
    this.doingOperations = false;
    this.isNumberPressed = false;
    this.template = `
    <div class="partners-calc-container">
            <div class="calc-display">
                <div class="calc-display-operation"></div>
                <div class="calc-display-result">0</div>
            </div>
            <div class="buttons-row">
                <div class="action-button" data-key="Delete">CE</div>
                <div class="action-button" data-key="Backspace"><i class="fas fa-long-arrow-alt-left"></i></div>
                <div class="action-button" data-key="%"><i class="fas fa-percentage"></i></div>
                <div class="action-button" data-key="/">/</div>
            </div>
            <div class="buttons-row">
                <div class="number-button" data-key="7">7</div>
                <div class="number-button" data-key="8">8</div>
                <div class="number-button" data-key="9">9</div>
                <div class="action-button" data-key="*"><i class="fas fa-times"></i></div>
            </div>
            <div class="buttons-row">
                <div class="number-button" data-key="4">4</div>
                <div class="number-button" data-key="5">5</div>
                <div class="number-button" data-key="6">6</div>
                <div class="action-button" data-key="-"><i class="fas fa-minus"></i></div>
            </div>
            <div class="buttons-row">
                <div class="number-button" data-key="1">1</div>
                <div class="number-button" data-key="2">2</div>
                <div class="number-button" data-key="3">3</div>
                <div class="action-button" data-key="+"><i class="fas fa-plus"></i></div>
            </div>
            <div class="buttons-row">
                <div class="extended_div number-button" data-key="0">0</div>
                <div class="number-button" data-key=".">.</div>
                <div class="action-button" data-key="Enter"><i class="fas fa-equals"></i></div>
            </div>
        </div>
    `;

    this.initCalc();

    this._outsideClickProxy = $.proxy(function (e) {
      this.outsideClick(e);
    }, this);

    // Bind global mousedown for hiding
    $(document)
      .on('mousedown', this._outsideClickProxy)
      // also support mobile devices
      .on('touchend', this._outsideClickProxy)
      // and also close when focus changes to outside the picker (eg. tabbing between controls)
      .on('focusin', this._outsideClickProxy);

    /** Event Listeners **/
    $(this.el).find('.partners-calc-container')
      .on('click', $.proxy(this.setActive, this));

    $(this.el).find('.partners-calc-container .buttons-row div')
      .on('click', $.proxy(this.handleClick, this));

    $(document)
      .on('keydown', $.proxy(this.handleKeyDown, this));
  };

  PartnersCalc.prototype = {
    constructor: PartnersCalc,

    /**
     * Inicia la calculadora insertando el HTML en el elemento que se inicializó el plugin.
     */
    initCalc: function () {
      $(this.el).html(this.template);
    },

    /**
     * Muestra la calculadora
     */
    show: function () {
      $(this.el).find('.partners-calc-container').show();
    },

    /**
     * Oculta la calculadora
     */
    hide: function () {
      $(this.el).find('.partners-calc-container').hide();
    },

    /**
     * Agregar la clase active para indicar que la calculadora se está usando
     */
    setActive: function () {
      if (!this.isActive) {
        $(this.el).find('.partners-calc-container').addClass('active');
        this.isActive = true;
      }
    },

    /**
     * Quita la clase de active de la caluladora, haciendo que esta ya no sea "funcional" sino hasta que la
     * vuelvan a actilet.
     * @param e
     */
    outsideClick: function (e) {
      let target = $(e.target);
      if (
        e.type === "focusin" ||
        target.closest(this.element).length ||
        target.hasClass('partners-calc-container') ||
        target.parents('div').hasClass('partners-calc-container') ||
        target.closest('div').hasClass('partners-calc-container')
      ) return;
      this.isActive = false;
      $(this.el).find('.partners-calc-container').removeClass('active');
    },

    /**
     * Agrega una pequeña y rápida animación para hacer aún más notorio el hecho de que acaban de dar un click.
     * @param e
     */
    animateClick: function (e) {
      let target = $(e);
      target.css('transform', 'scale(1.05)');
      setTimeout(function () {
        target.css('transform', 'scale(1)');
      }, 100);
    },

    /**
     * Controla los clicks que se den sobre cada botón de la calculadora, ya sea número o función.
     * @param e
     */
    handleClick: function (e) {
      let $this;
      if ($(e.target).is('div')) {
        $this = $(e.target);
      } else {
        $this = $(e.target).closest('div');
      }
      let value = $this.attr('data-key');
      if ($this.hasClass('number-button')) {
        if (value == '.') {
          this.handleDecimalPointPressed();
        } else {
          this.handleNumberPressed(value);
        }
      } else if ($this.hasClass('action-button')) {
        if (value === 'Backspace') {
          this.handleBackspacePressed();
        } else if (value === 'Delete') {
          this.handleDeletePressed();
        } else if (value === '+') { // Suma
          this.handleOperacion('+');
        } else if (value === '-') { // Resta
          this.handleOperacion('-');
        } else if (value === '*') { // Multiplicación
          this.handleOperacion('*');
        } else if (value === '/') { // División
          this.handleOperacion('/');
        } else if (value === 'Enter') {
          this.handleEnterKey();
        }
      }
      this.animateClick($this);
    },

    /**
     * Se encarga básicamente detectar todas las teclas y decidir que es lo que se tiene que realizar.
     * @param e
     */
    handleKeyDown: function (e) {
      // Solamente deberá funcionar cuando la calculadora este activa.
      if (this.isActive) {
        let keyCode = e.keyCode;
        let value = numeral(e.key).value();
        // Manejar los números, incluyendo los del numpad, y aparte el punto decimal.
        if (((keyCode >= 48 && keyCode <= 57) || (keyCode >= 96 && keyCode <= 105)) && !e.shiftKey) { // Números
          this.handleNumberPressed(value);
        } else if (keyCode === 8) { // Borrar
          this.handleBackspacePressed();
        } else if (keyCode === 46) { // Delete
          this.handleDeletePressed();
        } else if ((keyCode === 110 || keyCode === 190) && !e.shiftKey) { // Punto decimal
          this.handleDecimalPointPressed();
        } else if (e.key === '+') { // Suma
          this.handleOperacion('+');
        } else if (e.key === '-') { // Resta
          this.handleOperacion('-');
        } else if (e.key === '*') { // Multiplicación
          this.handleOperacion('*');
        } else if (e.key === '/') { // División
          this.handleOperacion('/');
        } else if (e.key === 'Enter') {
          this.handleEnterKey();
        }

        // Al final de tod0, agregar una animación en la calculadora
        // a las teclas que fueron "activadas/presionadas".
        this.animateKeyDown(e.key);
      }
    },

    /**
     * Controla t0do lo que tenga que ver con números con el fin de alterar el resultado.
     * @param {int | float} value
     */
    handleNumberPressed: function (value) {
      let newVal;


      // Si el ultimo valor de listaOperaciones es un signo, entonces deberá de actuar diferente, ya que
      // se espera que se borre el display para que el número que se este ingresando sea evaluado contra
      // la función del signo que se encontró y su valor anterior.
      // Ej, si la listaOperaciones = [1,'+'], como el ultimo elemento es un signo, se espera que el siguiente
      // numero que ponga se sume más el 1 que está en la listaOperaciones.
      if (this.listaOperaciones.length > 0) {
        if (this.isSymbol(this.listaOperaciones[this.listaOperaciones.length - 1])) {
          // Marcar el estatus de que se está haciendo una operación, de esta manera se podrá escribir libremente
          // en el result otra vez.
          if (!this.doingOperations) {
            // Limpiar el display
            newVal = value.toString();
            this.doingOperations = true;
          } else {
            // Ya se limpio previamente el display, ahora solo agregar el valor al display
            newVal = this.displayActiveValue.toString() + value.toString();
          }
        }
      } else {
        newVal = this.displayActiveValue.toString() + value.toString();
      }

      this.isNumberPressed = true;

      // Un "Bug" que ocurre, es que cuando llega a números muy grandes, y js lo representa con
      // notación exponencial, el siguiente format me romperia todos los expontentes y aparte el valor
      // lo resetea, no considero importante arreglarlo actualmente porque no creo que sea usada
      // para numeros grandes.

      let formattedVal = this.formatNumber(newVal);

      this.handleDisplayResult(formattedVal);
    },

    /**
     * Controla el comportamiento que se deberá de tener cuando se agregue un punto decimal.
     */
    handleDecimalPointPressed: function () {
      let newVal;
      if (this.listaOperaciones.length > 0) {
        if (this.isSymbol(this.listaOperaciones[this.listaOperaciones.length - 1])) {
          // Marcar el estatus de que se está haciendo una operación, de esta manera se podrá escribir libremente
          // en el result otra vez.
          if (!this.doingOperations) {
            // Limpiar el display
            newVal = "0.";
            this.doingOperations = true;

            // Deberá agregar el punto al final de lo que hay actualmente en el display.
            this.handleDisplayResult(newVal);
          } else {
            // Solamente trabajar cuando no se tenga ningún punto decimal, para evitar agregar varios.
            if (this.displayActiveValue.toString().indexOf('.') === -1) {
              // Ya se limpio previamente el display, ahora solo agregar el valor al display
              newVal = this.displayActiveValue.toString() + ".";

              // Deberá agregar el punto al final de lo que hay actualmente en el display.
              this.handleDisplayResult(newVal);
            }
          }
        }
      } else {
        // Solamente trabajar cuando no se tenga ningún punto decimal, para evitar agregar varios.
        if (this.displayActiveValue.toString().indexOf('.') === -1) {
          // Ya se limpio previamente el display, ahora solo agregar el valor al display
          newVal = this.displayActiveValue.toString() + ".";

          // Deberá agregar el punto al final de lo que hay actualmente en el display.
          this.handleDisplayResult(newVal);
        }
      }
    },

    /**
     * Controla el borrado de lo que se encuentra actualmente en el display.
     */
    handleBackspacePressed: function () {
      // Deberá de borrar el último caracter visible que se encuentra en el display
      let newVal = this.displayActiveValue.toString().slice(0, -1);
      let tmpVal = this.formatNumber(newVal);
      this.handleDisplayResult(tmpVal);
    },

    /**
     * Controla el resultado y la cola de operaciones cada vez que se realice una operación.
     * @param {string} operator
     */
    handleOperacion: function (operator) {
      // Debe de agregar lo que hay en displayActiveValue a la cola y aparte también agregar el simolo de más
      if (this.isNumberPressed) {
        // Si el ultimo elemento de la listaOperaciones es un simoblo, deberá de realizar las operaciones
        // y el resultado ponerlo en el display.
        if (this.listaOperaciones.length > 1) {
          if (this.isSymbol(this.listaOperaciones[this.listaOperaciones.length - 1])) {
            this.listaOperaciones.push(numeral(this.displayActiveValue).value());

            // Agregar el Operador correspondiente
            this.listaOperaciones.push(operator);
            this.doingOperations = false;
            this.isNumberPressed = false;

            // Calcular el total de tod0 lo que hay en la lista de operaciones.
            let tmpNewVal = this.calcularListaOperaciones();

            // Mostrar el total en el display
            this.handleDisplayResult(this.formatNumber(tmpNewVal.toString()));
          }
        } else {
          this.listaOperaciones.push(numeral(this.displayActiveValue).value());
          this.listaOperaciones.push(operator);

        }
        this.handleDisplayOperations();
      }
    },

    /**
     * Controla la tecla de Enter, que será utilizada para mostrar el resultado final de tod0 lo que haya en la lista
     * de operaciones y lo de que está en el display result.
     */
    handleEnterKey: function () {
      let resultListaOperaciones = this.calcularListaOperaciones();
      let result = 0;
      if (this.listaOperaciones[this.listaOperaciones.length - 1] === '+') {
        result = resultListaOperaciones + numeral(this.displayActiveValue).value();
      } else if (this.listaOperaciones[this.listaOperaciones.length - 1] === '-') {
        result = resultListaOperaciones - numeral(this.displayActiveValue).value();
      } else if (this.listaOperaciones[this.listaOperaciones.length - 1] === '*') {
        result = resultListaOperaciones * numeral(this.displayActiveValue).value();
      } else if (this.listaOperaciones[this.listaOperaciones.length - 1] === '/') {
        result = resultListaOperaciones / numeral(this.displayActiveValue).value();
      }

      this.handleDisplayResult(this.formatNumber(result.toString()));
      this.listaOperaciones = [];
      this.doingOperations = false;
      this.handleDisplayOperations();
    },

    /**
     * Elimina todo0 lo que este en el display y vacía la cola de operaciones
     */
    handleDeletePressed: function () {
      this.handleDisplayResult('0');
      this.listaOperaciones = [];
      this.doingOperations = false;
      this.handleDisplayOperations();
    },

    /**
     * Calcula tod0 lo que tenga la lista de operaciones en el orden en que se fueron ingresando los elementos de la lista.
     * @returns {int | float}
     */
    calcularListaOperaciones: function () {

      if (this.listaOperaciones.length === 0) return 0;
      // Se inicializa con el primer valor que tiene la lista de operaciones, que forzosamente tiene que ser un numero
      let result = this.listaOperaciones[0];

      let limit;
      if (this.isSymbol(this.listaOperaciones[this.listaOperaciones - 1])) {
        limit = this.listaOperaciones.length - 1;
      } else {
        limit = this.listaOperaciones.length;
      }

      // El ciclo empieza desde 1 para no tomar en cuenta el primer elemento de la lista, que seria el result que
      // anteriormente se inicializó.
      for (let i = 1; i < limit; i++) {
        if (this.isSymbol(this.listaOperaciones[i]) && this.listaOperaciones[i + 1] !== undefined) {
          if (this.listaOperaciones[i] === '+') {
            result = result + this.listaOperaciones[i + 1];
          } else if (this.listaOperaciones[i] === '-') {
            result = result - this.listaOperaciones[i + 1];
          } else if (this.listaOperaciones[i] === '*') {
            result = result * this.listaOperaciones[i + 1];
          } else if (this.listaOperaciones[i] === '/') {
            result = result / this.listaOperaciones[i + 1];
          }
        }
      }

      return result;
    },


    /**
     * Compara un valor y determina si es un simbolo.
     * @param {string} value
     * @returns {boolean}
     */
    isSymbol: function (value) {
      return value === '+' || value === '-' || value === '*' || value === '/';
    },


    /**
     * Formatea el número que se pase como parametro, agregando los decimales que se necesitan (en caso de ser necesarios)
     * @param {string} number
     * @return {string} formattedNumber
     */
    formatNumber: function (number) {
      let formattedNumber = 0;
      // Si tiene un punto, calcular el total de decimales a mostrar
      if (number.indexOf('.') > -1) {
        let tmp = number.split('.');
        let decimals = tmp[tmp.length - 1].length;
        let formatString = '0,0.';
        for (let i = 0; i < decimals; i++) {
          formatString += '0';
        }

        formattedNumber = numeral(number).format(formatString);
      } else {
        // No tiene punto decimal, solamente formatear miles en caso de ser necesario.
        formattedNumber = numeral(number).format('0,0');
      }

      return formattedNumber;
    },

    /**
     * Controla la cola de operaciones que se han realizado y las muestra en el display
     */
    handleDisplayOperations: function () {
      // Básicamente lo que tiene que hacer está función, es vaciar la lista de operaciones y mostrarla
      // en el display.
      let listString = '';
      for (let i = 0; i < this.listaOperaciones.length; i++) {
        listString += this.listaOperaciones[i] + ' ';
      }
      $(this.el).find('.calc-display-operation').html(listString);
    },

    /**
     * Se encarga de mostrar el resultado en el display de la calculadora, y aparte controla que este
     * mismo no se desborde.
     * @param {string} value
     */
    handleDisplayResult: function (value) {
      // Reajustar el tamaño de la fuente para que siempre se mantenga dentro del display
      if (value.length >= 14 && value.length <= 15) {
        $(this.el).find('.calc-display-result').css({'font-size': '30px', 'padding-top': '5px'});
      } else if (value.length > 15 && value.length <= 17) {
        $(this.el).find('.calc-display-result').css({'font-size': '27px', 'padding-top': '5px'});
      } else if (value.length >= 18 && value.length <= 20) {
        $(this.el).find('.calc-display-result').css({'font-size': '21px', 'padding-top': '10px'});
      } else if (value.length >= 21) {
        $(this.el).find('.calc-display-result').css({'font-size': '17px', 'padding-top': '10px'});
      } else {
        $(this.el).find('.calc-display-result').removeAttr('style');
      }

      $(this.el).find('.calc-display-result').html(value);
      this.displayActiveValue = value;
    },

    /**
     * Regresa el total actual que existe en el display de la calculadora
     * @returns {*|int|float}
     */
    getResultCalc: function () {
      let result = numeral(this.displayActiveValue).value();
      return result;
    },

    /**
     * Agrega una animación al teclado de la calculadora cuando los números sean presionados mediante el teclado
     * fisico y no mediante la calculadora, para mostrar cuales teclas se están "activando"
     * @param keyValue
     */
    animateKeyDown: function (keyValue) {
      let el = $(this.el).find('.partners-calc-container .buttons-row div[data-key="' + keyValue + '"]');
      if (el) {
        el.addClass('active');
        setTimeout(function () {
          el.removeClass('active');
        }, 100);
      }
    }
  };

  $.fn.partnersCalc = function (options) {
    if (options === undefined || typeof options === 'object') {
      // Creates a new plugin instance, for each selected element, and
      // stores a reference withint the element's data
      return this.each(function () {
        if (!$.data(this, 'partnersCalc')) {
          $.data(this, 'partnersCalc', new PartnersCalc(this, options));
        }
      });
    } else if (typeof options === 'string') {
      // If the user does not pass any arguments and the method allows to
      // work as a getter then break the chainability so we can return a value
      // instead the element reference.
      let instance = $.data(this[0], 'partnersCalc');
      return instance[options].apply(instance, Array.prototype.slice.call(options, 1));
    }

  };

  return PartnersCalc;
}));