<?php

use Defuse\Crypto\Key;
use Mpdf\Mpdf;

class sys_tools
{

  public function __construct($tipo, $data)
  {
    if (!$data) {
      $this->throw_message('error', 'No data');
    } else {
      switch ($tipo) {
        case 'report_error':
          try {
            $this->report_error($data);
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'report_fatal_error');
          }
          break;
        case 'inline_edit':
          try {
            $this->inline_edit($data);
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'edicion_inline');
          }
          break;
        case 'user_list':
          try {
            $this->user_list();
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'user_list');
          }
          break;
        case 'load_obs':
          try {
            $this->load_obs($data);
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'cargar_observaciones');
          }
          break;
        case 'load_producto_info_mercancia':
          try {
            $this->load_producto_info($data);
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'load_producto_info_mercancia');
          }
          break;
        case 'vista_factura_prev':
          try {
            $this->vista_factura_prev($data);
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'vista_factura_prev');
          }
          break;
        case 'get_datos_cliente_contpaq':
          try {
            $this->get_datos_cliente_contpaq($data);
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'get_datos_cliente_contpaq');
          }
          break;
        case 'validate_factura_pagada':
          try {
            $this->validate_factura_pagada($data);
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'validate_factura_pagada');
          }
          break;
        case 'get_full_factura':
          try {
            $this->get_full_factura($data);
          } catch (Exception $e) {
            $this->throw_fatal_error($e, 'get_full_factura');
          }
          break;
        default:
          break;
      }
    }
  }

  protected function throw_message($status, $msg)
  {
    echo json_encode(
      array(
        'status' => $status,
        'message' => $msg
      )
    );
  }

  private function report_error($data)
  {
    db_insert('partners_error_reports')
      ->fields(
        array(
          'user_report' => $data->err_user_id,
          'report_date' => $data->err_time,
          'report_location' => $data->err_loc,
          'report_msg' => $data->err_msg
        )
      )
      ->execute();

    $this->register_log_actions('partners_error_reports', 'fatal_error', $data->err_msg);
    $this->throw_message('success', $data);
  }

  /**
   * @param $table {string} a la cual se hace la consulta
   * @param $type {string} ya sea update,insert,delete
   * @param $message {string} breve descripción de la consulta
   * @throws Exception
   */
  protected function register_log_actions($table, $type, $message)
  {
    global $user, $base_root;
    db_insert('partners_log_actions')
      ->fields(
        array(
          'timestamp' => REQUEST_TIME,
          'uid' => $user->uid,
          'hostname' => ip_address(),
          'location' => $base_root . request_uri(),
          'table_at' => $table,
          'type' => $type,
          'message' => $message
        )
      )
      ->execute();
  }

  protected function throw_fatal_error($e, $location)
  {
    global $user;
    $this->throw_message('error', 'Lamentamos lo sucedido, ya se ha enviado el reporte del error para que sea atendido lo más pronto posible!');
    $id_report = db_insert('partners_error_reports')
      ->fields(
        array(
          'user_report' => $user->uid,
          'report_date' => REQUEST_TIME,
          'report_location' => $location,
          'report_msg' => $e->getMessage(),
          'detailed_description' => json_encode($e->getTrace())
        )
      )
      ->execute();

    $transport = new Swift_SendmailTransport();
    $mailer = new Swift_Mailer($transport);
    $from = ['errores@pcpartners.com.mx' => 'Pc Partners - Error Reports'];
    $to = ['omar@pcpartners.com.mx' => 'Pc Partners - Omar', 'roberto@pcpartners.com.mx' => 'Pc Partners - Roberto'];
    $data['id_report'] = $id_report;
    $data['message'] = $e->getMessage();
    $data['trace'] = $e->getTrace();
    $body = render_template('php', 'system.mail_send_report', $data);
    $message = (new Swift_Message())
      ->setSubject('Reporte de error fatal en PcPartners: ' . $id_report)
      ->setFrom($from)
      ->setBody($body, 'text/html');
    $failedRecipients = [];
    $numSent = 0;

    foreach ($to as $address => $name) {
      if (is_int($address)) {
        $message->setTo($name);
      } else {
        $message->setTo([$address => $name]);
      }
      $this->register_log_actions('partners_error_reports', 'mail', 'Se envió automáticamente el reporte de un error fatal a Erick.');
      $numSent += $mailer->send($message, $failedRecipients);
    }
  }

  /**
   * @param $data : Datos necesarios utilizados para editar un campo en una tabla
   * @throws Exception
   */
  private function inline_edit($data)
  {
    global $user;
    if ($data->type == 'date') {
      $data->new_val = strtotime($data->new_val);
    }

    if (is_array($data->new_val)) {
      $data->new_val = implode(',', $data->new_val);
    }

    //Actualizar el correo principal también
    if ($data->edit_table == "partners_clientes_datos") {
      $datoCliente = db_select("partners_clientes_datos", "pcd")
        ->fields("pcd", array('tipo_dato', 'id_cliente'))
        ->condition($data->edit_table_key, $data->edit_id)
        ->execute()
        ->fetchAssoc();
      if ((int) $datoCliente['tipo_dato'] == 3) {
        db_update("partners_clientes")
          ->fields(
            array(
              'email' => $data->new_val
            )
          )
          ->condition('id_cliente', $datoCliente['id_cliente'])
          ->execute();
      }
    }
    //.

    db_update($data->edit_table)
      ->fields(
        array(
          $data->edit_col => $data->new_val,
          'updated_at' => REQUEST_TIME,
          'updated_by' => $user->uid,
        )
      )
      ->condition($data->edit_table_key, $data->edit_id)
      ->execute();

    $this->register_log_actions($data->edit_table, 'update', 'Se editó el dato ' . $data->edit_col . ' con el valor: ' . $data->new_val);
    $this->throw_message('success', 'Cambios guardados.');
  }

  private function user_list()
  {
    $role = user_role_load_by_name('Todos');
    $query = 'SELECT ur.uid
        FROM {users_roles} AS ur
        WHERE ur.rid = :rid';
    $result = db_query($query, array(':rid' => $role->rid));
    $uids = $result->fetchCol();

    $usuarios = db_select('users', 'u')
      ->fields('u', array('uid', 'name'))
      ->condition('status', 1)
      ->condition('uid', $uids, 'IN')
      ->condition('uid', 26, '!=')
      ->orderBy('name', 'ASC')
      ->execute()->fetchAll(PDO::FETCH_ASSOC);

    $users = $usuarios;


    $users_in_role = array();
    foreach ($users as $row) {
      $users_in_role[] = array(
        'uid' => $row['uid'],
        'name' => $row['name']
      );
    }

    $this->throw_message('success', json_encode($users_in_role));
  }

  /**
   * @param $datos ->type 1 : Load obs, 2: Capturar nueva
   * @throws Exception
   */
  private function load_obs($datos)
  {
    global $user;
    if ($datos->type == 1) {
      $get_obs = db_select($datos->tabla, 'custom_alias')
        ->fields('custom_alias', array($datos->col));
      if ($datos->custom_condition) {
        $get_obs->where($datos->custom_condition);
      } else {
        $get_obs->condition($datos->index, $datos->id_index);
      }


      $get_observaciones = $get_obs->execute()->fetchField();

      $obs['observaciones'] = unserialize($get_observaciones);
      $obs['only_view'] = (isset($datos->only_view) && $datos->only_view != '') ? $datos->only_view : false;
      $obs['allowReminder'] = (isset($datos->allowReminder) && $datos->allowReminder != '') ? $datos->allowReminder : false;
      //var_dump($obs['observaciones']);
      $vista = render_template('php', 'system.load_obs', $obs);
      $this->throw_message('success', $vista);
    } else if ($datos->type == 2) {
      # Si trae para capturar recordatorio, solo capturar el recordatorio porque la observación ya se debería de haber capturado.
      if (isset($datos->setReminder) && $datos->setReminder != '') {
        $id_recordatorio = db_insert('partners_pendientes')
          ->fields(
            array(
              'created_at' => REQUEST_TIME,
              'created_by' => $user->uid,
              'updated_at' => REQUEST_TIME,
              'updated_by' => $user->uid,
              'tipo_pendiente' => 1,
              'area_pendiente' => 5,
              # Por default lo pongo en Admon, sin importar el area del usuario
              'id_cliente' => $datos->idCliente,
              'asignado_para' => $user->uid,
              'fecha_pendiente' => strtotime($datos->fechaReminder),
              'descripcion_pendiente' => "Asunto: " . $datos->subjectReminder . " | " . $datos->obsReminder,
              'horario_pendiente' => $datos->horaReminder,
              'quien_reporta_pendiente' => $user->name,
              'prioridad_pendiente' => 1 # Entra en urgente automáticamente
            )
          )
          ->execute();

        $this->register_log_actions('partners_pendientes', 'insert', "Se capturó un recordatorio de llamada ($id_recordatorio)");
        $this->throw_message('success', 'Bien!');
      } else {
        $data = $this->serialize_obs($datos->new_val, 2, $datos->tabla, $datos->index, $datos->id_index, $datos->name, $datos->col);
        if (isset($datos->extra_val) && $datos->extra_val != '') {
          // Adaptación para cobranza
          $check = db_select($datos->tabla, 'c')
            ->fields('c', array($datos->index));
          if ($datos->custom_condition) {
            $check->where($datos->custom_condition);
          } else {
            $check->condition($datos->index, $datos->id_index);
          }
          $check_ = $check->execute()->fetchField();

          if (!$check_) {
            db_insert($datos->tabla)
              ->fields(
                array(
                  $datos->col => $data,
                  $datos->extra_col => $datos->extra_val
                )
              )
              ->execute();
          } else {
            $update = db_update($datos->tabla)
              ->fields(
                array(
                  $datos->col => $data,
                  $datos->extra_col => $datos->extra_val
                )
              );
            if ($datos->custom_condition) {
              $update->where($datos->custom_condition);
            } else {
              $update->condition($datos->index, $datos->id_index);
            }
            $update->execute();
          }
        } else {
          db_update($datos->tabla)
            ->fields(
              array(
                $datos->col => $data
              )
            )
            ->condition($datos->index, $datos->id_index)
            ->execute();
        }
        $date = new DateTime();
        $date = $date->format("d/m/y H:i:s");
        $this->throw_message('success', '<tr><td>' . $date . '</td><td>' . ((isset($datos->name)) ? $datos->name : $user->name) . '</td><td>' . $datos->new_val . '</td></tr>');
      }
    }
  }

  /**
   * @param $obs {string} texto normal de observaciones
   * @param null $table {string} a la cual hacer la consulta ( solo tipo 2 )
   * @param null $index {string} de la tabla ( solo tipo 2 )
   * @param null $id_reg {string} id_principal de la tabla ( solo tipo 2 )
   * @param $type {string/int} 1: solo capturar, 2: actualizar observaciones ( trae las existentes y le agrega la nueva )
   * @param $user_auth {string} Cuando != null es el nombre del usuario pasado mediante auth_user
   * @param $col_name {string} Cuando el nombre de la columna no es observaciones, se puede pasar el nombre que si es.
   * @return string lo que se debe de ingresar a la tabla ya formateado.
   */
  protected function serialize_obs($obs, $type, $table = NULL, $index = NULL, $id_reg = NULL, $user_auth = NULL, $col_name = 'observaciones')
  {
    global $user;
    $usuario = $user_auth ? $user_auth : $user->name;
    $date = new DateTime();
    $date = $date->format("d/m/y H:i:s");
    if ($type == 1) {
      $dato = array(
        "fecha" => $date,
        "usuario" => $usuario,
        "texto" => $obs,
      );
      $data[] = $dato;
    } else if ($type == 2) {
      $query = db_select($table, 'custom_alias');
      $query->fields('custom_alias', array($col_name))
        ->condition('custom_alias.' . $index . '', $id_reg);
      $result = $query->execute();
      $record = $result->fetchAssoc();
      $anterior = unserialize($record[$col_name]);
      $dato = array(
        "fecha" => $date,
        "usuario" => $usuario,
        "texto" => $obs,
      );
      $anterior[] = $dato;
      $data = $anterior;
    }

    return serialize($data);
  }

  /**
   * @param $datos ->type 1: traer nombre_producto, 2: traer costo producto
   */
  protected function load_producto_info($datos)
  {
    if ($datos->val != "") {
      $access_credentials = db_select('partners_sql_connections', 'psc')
        ->fields('psc')
        ->execute()->fetchAssoc();
      $user_access = self::encrypt_decrypt('decrypt', $access_credentials['user']);
      $pass_access = self::encrypt_decrypt('decrypt', $access_credentials['pass']);
      $server_access = self::encrypt_decrypt('decrypt', $access_credentials['server_ip']);
      $server_db_access = self::encrypt_decrypt('decrypt', $access_credentials['server_db']);

      //$objConnect = mssql_connect($server_access, $user_access, $pass_access);
      //mssql_select_db($server_db_access);

      $objConnect = sqlsrv_connect($server_access, ["Database" => $server_db_access, "UID" => $user_access, "PWD" => $pass_access]);

      if ($objConnect) {
        if ($datos->type == 1) {
          $query = "
                      SELECT
                      admProductos.CNOMBREPRODUCTO AS 'PRODUCTO'
                      FROM admProductos
                      WHERE (admProductos.CCODIGOPRODUCTO='$datos->val')
                    ";
          // $result = mssql_query($query);
          $result = sqlsrv_query($objConnect, $query);
          $out = false;
          //while ($obj = mssql_fetch_object($result)) {
          while ($obj = sqlsrv_fetch_object($result)) {
            $out = $obj->PRODUCTO;
          }
          $out = utf8_encode($out);
          if ($out != false) {
            $this->throw_message('success', $out);
          } else {
            $this->throw_message('error', 'No se encontró ningún producto con este UPC.');
          }
        } else if ($datos->type == 2) {
          $query = "
                      SELECT
                      admMovimientos.CPRECIO as costo
                      FROM
                      adLRNV2016.dbo.admDocumentos admDocumentos,
                      adLRNV2016.dbo.admMovimientos admMovimientos,
                      adLRNV2016.dbo.admMovimientosSerie admMovimientosSerie,
                      adLRNV2016.dbo.admNumerosSerie admNumerosSerie,
                      adLRNV2016.dbo.admProductos admProductos
                      WHERE
                      admMovimientos.CIDDOCUMENTO = admDocumentos.CIDDOCUMENTO
                      AND admMovimientos.CIDDOCUMENTODE  = 19
                      AND admProductos.CIDPRODUCTO = admMovimientos.CIDPRODUCTO
                      AND admNumerosSerie.CIDPRODUCTO = admProductos.CIDPRODUCTO
                      AND admMovimientosSerie.CIDMOVIMIENTO = admMovimientos.CIDMOVIMIENTO
                      AND admMovimientosSerie.CIDSERIE = admNumerosSerie.CIDSERIE
                      AND ((admNumerosSerie.CNUMEROSERIE='$datos->val'))
                    ";
          // $result = mssql_query($query);
          $result = sqlsrv_query($objConnect, $query);
          $out = false;
          // while ($obj = mssql_fetch_object($result)) {
          while ($obj = sqlsrv_fetch_object($result)) {
            $out = $obj->costo;
          }


          if ($out != false) {
            $this->throw_message('success', $out);
          } else {
            $this->throw_message('error', 'No se encontró ningún costo con este número de Serie.');
          }
        }
      } else {
        $this->throw_message('error', 'No se puede conectar a la base de datos');
        // die( print_r( sqlsrv_errors(), true));
      }
    }
  }

  public static function encrypt_decrypt($action, $string)
  {
    $output = false;
    $encrypt_method = "AES-256-CBC";
    $secret_key = 'V5fRcLWZBixSD5hG5h5J';
    $secret_iv = 'zRßän»Èÿp¢Ã¦X';
    // hash
    $key = hash('sha256', $secret_key);

    // iv - encrypt method AES-256-CBC expects 16 bytes - else you will get a warning
    $iv = substr(hash('sha256', $secret_iv), 0, 16);
    if ($action == 'encrypt') {
      $output = openssl_encrypt($string, $encrypt_method, $key, 0, $iv);
      $output = base64_encode($output);
    } else if ($action == 'decrypt') {
      $output = openssl_decrypt(base64_decode($string), $encrypt_method, $key, 0, $iv);
    }
    return $output;
  }

  protected function vista_factura_prev($numero_factura)
  {
    $numero_factura = trim($numero_factura);
    $access_credentials = db_select('partners_sql_connections', 'psc')
      ->fields('psc')
      ->execute()->fetchAssoc();
    $user_access = self::encrypt_decrypt('decrypt', $access_credentials['user']);
    $pass_access = self::encrypt_decrypt('decrypt', $access_credentials['pass']);
    $server_access = self::encrypt_decrypt('decrypt', $access_credentials['server_ip']);
    $server_db_access = self::encrypt_decrypt('decrypt', $access_credentials['server_db']);

    //$objConnect = mssql_connect($server_access, $user_access, $pass_access);
    //mssql_select_db($server_db_access);
    $objConnect = sqlsrv_connect($server_access, ["Database" => $server_db_access, "UID" => $user_access, "PWD" => $pass_access]);

    if ($objConnect) {
      $query = "
                  SELECT admDocumentos.CSERIEDOCUMENTO AS SERIE,
                    admDocumentos.CFOLIO,
                    admDocumentos.CIDAGENTE,
                    admDocumentos.CFECHA,
                    admDocumentos.CIDCLIENTEPROVEEDOR,
                    admDocumentos.CRAZONSOCIAL AS NOMBRE,
                    admProductos.CNOMBREPRODUCTO AS PRODUCTO,
                    admProductos.CCODIGOPRODUCTO,
                    admMovimientos.CUNIDADES,
                    admMovimientos.CPRECIO,
                    admMovimientos.CCOSTOESPECIFICO
                  FROM admDocumentos admDocumentos,
	                admMovimientos admMovimientos,
	                admProductos admProductos
                  WHERE admMovimientos.CIDDOCUMENTO = admDocumentos.CIDDOCUMENTO
                  AND admProductos.CIDPRODUCTO = admMovimientos.CIDPRODUCTO
                  AND ((admDocumentos.CIDDOCUMENTODE=4)
                  AND ( (admDocumentos.CIDCONCEPTODOCUMENTO=3018) OR (admDocumentos.CIDCONCEPTODOCUMENTO=3025) )
                  AND (admDocumentos.CFOLIO=$numero_factura))
                    ";
      //$result = mssql_query($query);
      $result = sqlsrv_query($objConnect, $query);
      //while ($obj = mssql_fetch_object($result)) {
      while ($obj = sqlsrv_fetch_object($result)) {
        $x = gettype($obj->CFECHA) == "string" ? new DateTime($obj->CFECHA) : $obj->CFECHA;
        $f = $x->getTimestamp();

        $factura[] = array(
          "data" => array(
            $obj->SERIE,
            $obj->CFOLIO,
            $obj->CIDAGENTE,
            date('d/m/y', $f),
            $obj->CIDCLIENTEPROVEEDOR,
            $obj->NOMBRE,
            $obj->PRODUCTO,
            $obj->CCODIGOPRODUCTO,
            array("data" => number_format($obj->CUNIDADES, 0, '.', ','), "class" => array("text-right")),
            array("data" => number_format($obj->CPRECIO, 2, '.', ','), "class" => array("text-right")),
            array("data" => number_format($obj->CCOSTOESPECIFICO, 2, '.', ','), "class" => array("text-right"))
          ),
        );
      }

      $header_cols = array(
        array("name" => "Serie", "sortable" => 0),
        array("name" => "Folio", "sortable" => 0),
        array("name" => "Agente", "sortable" => 0),
        array("name" => "Fecha", "sortable" => 0),
        array("name" => "Código Cliente", "sortable" => 0),
        array("name" => "Nombre", "sortable" => 0),
        array("name" => "Producto", "sortable" => 0),
        array("name" => "Código Producto", "sortable" => 0),
        array("name" => "Unidades", "sortable" => 0),
        array("name" => "Precio", "sortable" => 0),
        array("name" => "Costo", "sortable" => 0),
      );

      $header = sys_tools::sort_table("CFOLIO", "ASC", $header_cols);

      $table = array(
        "header" => $header,
        "rows" => $factura,
        "sticky" => false,
        "attributes" => array("class" => array("table-sm"))
      );


      $result = theme('table', $table);

      $this->throw_message('success', $result);
    } else {
      $this->throw_message('error', 'No se puede conectar a la base de datos');
      // die( print_r( sqlsrv_errors(), true));
    }
  }

  /**
   * @param $sort_by : por cual columna se esta filtrando
   * @param $order_by : ASC / DESC
   * @param $cols : todas las columnas a generar
   * @return array : $header utilizado para theme_table de drupal
   */
  public static function sort_table($sort_by, $order_by, $cols)
  {

    //Todo = Ver la forma de que tambien se pueda ordenar por campos de otras tablas

    $header = array();
    foreach ($cols as $col) {

      if ($col["sortable"] == 1) {
        if ($col["table_field"] == $sort_by) {
          if ($order_by == 'ASC') {
            $c = '<div class="cursor-pointer sortable_table_th" data-sort-by="' . $col["table_field"] . '" data-sort-order="ASC"> ' . $col["name"] . ' <i class="fa fa-sort-amount-asc float-right" aria-hidden="true"></i></div>';
          } else if ($order_by == 'DESC') {
            $c = '<div class="cursor-pointer sortable_table_th" data-sort-by="' . $col["table_field"] . '" data-sort-order="DESC"> ' . $col["name"] . ' <i class="fa fa-sort-amount-desc float-right" aria-hidden="true"></i></div>';
          }
        } else {
          $c = '<div class="cursor-pointer sortable_table_th" data-sort-by="' . $col["table_field"] . '" data-sort-order="ASC"> ' . $col["name"] . '<i class="fa fa-sort float-right" aria-hidden="true"></i> </div>';
        }
      } else {
        $c = $col['name'];
      }

      $header[] = $c;
    }

    return $header;
  }

  /**
   * Trae los datos del cliente en base al número de factura.
   * @param $datos
   */
  protected function get_datos_cliente_contpaq($datos)
  {
    $access_credentials = db_select('partners_sql_connections', 'psc')
      ->fields('psc')
      ->execute()->fetchAssoc();
    $user_access = self::encrypt_decrypt('decrypt', $access_credentials['user']);
    $pass_access = self::encrypt_decrypt('decrypt', $access_credentials['pass']);
    $server_access = self::encrypt_decrypt('decrypt', $access_credentials['server_ip']);
    $server_db_access = self::encrypt_decrypt('decrypt', $access_credentials['server_db']);

    //$objConnect = mssql_connect($server_access, $user_access, $pass_access);
    //mssql_select_db($server_db_access);
    $objConnect = sqlsrv_connect($server_access, ["Database" => $server_db_access, "UID" => $user_access, "PWD" => $pass_access]);

    if ($objConnect) {
      $query = "
              SELECT a.CRAZONSOCIAL AS 'CLIENTE',
                c.CCODIGOCLIENTE,
                C.CRFC
              FROM admDocumentos a
              LEFT OUTER JOIN admDomicilios b
              ON a.CIDCLIENTEPROVEEDOR = b.CIDCATALOGO
              LEFT JOIN admClientes c
              ON c.CIDCLIENTEPROVEEDOR = a.CIDCLIENTEPROVEEDOR
              WHERE (a.CFOLIO = '$datos->numero_factura')
              AND (a.CIDDOCUMENTODE=4)
              AND (a.CSERIEDOCUMENTO='E')
              AND (a.CCANCELADO=0)
              AND (b.CTIPOCATALOGO=1 OR b.CTIPOCATALOGO IS NULL)
                    ";
    } else {
      $this->throw_message('error', 'No se puede conectar a la base de datos');
      // die( print_r( sqlsrv_errors(), true));
    }

    // $result = mssql_query($query);
    $result = sqlsrv_query($objConnect, $query);
    $data = [];
    //while ($obj = mssql_fetch_object($result)) {
    while ($obj = sqlsrv_fetch_object($result)) {
      $data['nombre'] = utf8_encode($obj->CLIENTE);
      $data['codigo_cliente'] = $obj->CCODIGOCLIENTE;
      $data['rfc'] = $obj->CRFC;
    }
    if (!empty($data)) {
      $this->throw_message('success', json_encode($data));
    } else {
      $this->throw_message('error', 'No se encontró esa factura.');
    }
  }

  /**
   * @param $factura {int} a buscar el número de factura
   * @return integer
   */
  protected function validate_factura_pagada($factura)
  {
    $access_credentials = db_select('partners_sql_connections', 'psc')
      ->fields('psc')
      ->execute()->fetchAssoc();
    $user_access = self::encrypt_decrypt('decrypt', $access_credentials['user']);
    $pass_access = self::encrypt_decrypt('decrypt', $access_credentials['pass']);
    $server_access = self::encrypt_decrypt('decrypt', $access_credentials['server_ip']);
    $server_db_access = self::encrypt_decrypt('decrypt', $access_credentials['server_db']);

    // $objConnect = mssql_connect($server_access, $user_access, $pass_access);
    // mssql_select_db($server_db_access);
    $objConnect = sqlsrv_connect($server_access, ["Database" => $server_db_access, "UID" => $user_access, "PWD" => $pass_access]);

    if ($objConnect) {
      $query = "
              SELECT a.CPENDIENTE
              FROM admDocumentos a
              WHERE (a.CFOLIO = '$factura')
              AND (a.CIDDOCUMENTODE=4)
              AND (a.CCANCELADO=0)
              AND (a.CPENDIENTE>0)
              ORDER BY a.CFOLIO DESC
            ";

      //$result = mssql_query($query);
      $result = sqlsrv_query($objConnect, $query);
      $is_pagada = 0;
      //while ($obj = mssql_fetch_object($result)) {
      while ($obj = sqlsrv_fetch_object($result)) {
        $is_pagada++;
      }
    } else {
      $this->throw_message("error", "No se pudo conectar.");
      // die( print_r( sqlsrv_errors(), true));
    }


    return $is_pagada;
  }

  /**
   * Regresa el nombre del archivo .zip que tendrá la factura, consta del RFC del cliente y el folio de la factura.
   * @param $id_factura
   */
  protected function get_full_factura($id_factura)
  {
    // Poner en class.servicios en un data-factura esta funcion para regresar el nombre de factura, una vez con el nombre, se
    // usara para generar el pdf.
    $access_credentials = db_select('partners_sql_connections', 'psc')
      ->fields('psc')
      ->execute()->fetchAssoc();
    $user_access = self::encrypt_decrypt('decrypt', $access_credentials['user']);
    $pass_access = self::encrypt_decrypt('decrypt', $access_credentials['pass']);
    $server_access = self::encrypt_decrypt('decrypt', $access_credentials['server_ip']);
    $server_db_access = self::encrypt_decrypt('decrypt', $access_credentials['server_db']);

    //$objConnect = mssql_connect($server_access, $user_access, $pass_access);
    //mssql_select_db($server_db_access);
    $objConnect = sqlsrv_connect($server_access, ["Database" => $server_db_access, "UID" => $user_access, "PWD" => $pass_access]);

    if ($objConnect) {
      $query = "
                      SELECT
                        a.CSERIEDOCUMENTO AS 'SERIE',
                        a.CFOLIO AS 'FOLIO',
                        a.CFECHA AS 'FECHA',
                        a.CFECHAVENCIMIENTO AS 'VENCIMIENTO',
                        a.CTOTAL AS 'TOTAL',
                        a.CPENDIENTE AS 'SALDO',
                        a.CNUMEROGUIA AS 'NOTAS',
                        c.CCODIGOCLIENTE,
                        c.CRFC,
                        c.CEMAIL1,
                        c.CEMAIL2,
                        c.CEMAIL3
                      FROM admDocumentos a
                      LEFT OUTER JOIN admDomicilios b
                      ON a.CIDCLIENTEPROVEEDOR = b.CIDCATALOGO
                      LEFT JOIN admClientes c
                      ON c.CIDCLIENTEPROVEEDOR = a.CIDCLIENTEPROVEEDOR
                      WHERE (a.CFOLIO = '$id_factura')
                      AND (a.CIDDOCUMENTODE=4)
                      AND (a.CSERIEDOCUMENTO='E')
                      AND (a.CCANCELADO=0)
                      AND (b.CTIPOCATALOGO=1)
                    ";
    } else {
      $this->throw_message("error", "No se pudo conectar.");
      // die( print_r( sqlsrv_errors(), true));
    }

    //$result = mssql_query($query);
    $result = sqlsrv_query($objConnect, $query);
    $nombre = "not_on_server";
    //while ($obj = mssql_fetch_object($result)) {
    while ($obj = sqlsrv_fetch_object($result)) {
      $nombre = $obj->CRFC . 'FE00000' . $obj->FOLIO;
    }

    $check_exists = file_exists(HOME_SERVER . 'facturas/' . $nombre . '.zip');
    if (!$check_exists) {
      $this->throw_message('success', 'not_on_server');
    } else {
      $this->throw_message('success', $nombre);
    }
  }

  /**
   * @param $tipo , dependiendo de lo que quiera traer, algunos casos necesito todos los clientes, contactos,
   * proveedores, etc. y otras veces solo necesito los puros contactos
   */
  public static function lista_clientes($tipo)
  {
    $clientes = db_select('partners_clientes', 'pc')
      ->fields('pc', array("id_cliente", "nombre_cliente", "alias", "email", "nombre_facturacion", "pagina_web", "tipo_clasificacion", "credito", "tipo_cliente", "tipo_negocio", "saldo_horas"))
      ->condition('tipo_cliente', 1)
      ->condition('cancelado', 1, '!=')
      ->execute()->fetchAll(PDO::FETCH_ASSOC);
    $contactos = db_select('partners_clientes', 'pc')
      ->fields('pc', array("id_cliente", "nombre_cliente", "alias", "email", "nombre_facturacion", "pagina_web", "tipo_clasificacion", "credito", "tipo_cliente", "tipo_negocio", "saldo_horas"))
      ->condition('tipo_cliente', 2)
      ->condition('cancelado', 1, '!=')
      ->execute()->fetchAll(PDO::FETCH_ASSOC);
    $proveedores = db_select('partners_clientes', 'pc')
      ->fields('pc', array("id_cliente", "nombre_cliente", "alias", "email", "nombre_facturacion", "pagina_web", "tipo_clasificacion", "credito", "tipo_cliente", "tipo_negocio", "saldo_horas"))
      ->condition('tipo_cliente', 3)
      ->condition('cancelado', 1, '!=')
      ->execute()->fetchAll(PDO::FETCH_ASSOC);


    // *** Al momento de traer a los proveedores, traerlos con todoo y sus contactos. ***


    if ($tipo == "complete") {
      echo json_encode(
        array(
          "data" => array(
            "datos" => array(
              "clientes" => $clientes,
              "contactos" => $contactos,
              "proveedores" => $proveedores
            )
          )
        )
      );
    }
  }

  /**
   * SOLAMENTE ES UTILIZADO EN EL REPORTE DE CLIENTES, POR ESO HA SIDO MODIFICADO ESPECIFICAMENTE PARA EL REPORTE
   * DE CLIENTES, NO USAR EN NINGÚN OTRO LUGAR.
   * @param $options array de opciones
   * @param $query DatabaseStatementInterface  query a ejectuar para sacar el total de rows
   * @return array; [0] = $offset, [1] = $limit, [2] = $rendered_pagination
   */
  public static function pagination($options, $query)
  {
    /**
     *
     *  PARA USO EXCLUSIVO DEL REPORTE DE CLIENTES, ESTA PERSONALIZADO SOLO PARA USAR CON CLIENTES, NO USAR
     *  EN OTRO LADO.
     *
     */
    /*
        $count = db_select($options['table'], $options['alias']);
        $count->addExpression("COUNT($options[index])");
        $where = (isset($options['where']) && $options['where'] != '') ? $options['where'] : false;
        $filter = $options['filter'];
        if ($where) {
            $count->condition($where['field'], $where['condition'], $where['operator']);
        }

        if (isset($options['deleted_rows']) && $options['deleted_rows'] != '') {
            $count->condition($options['deleted_rows'], 1, '!=');
        }


        if ($filter) {
            foreach ($filter as $filtro) {
                $type = $filtro['type'];
                switch ($type) {
                    case 'condition':
                        $count->condition($filtro['condition'], $filtro['value'], $filtro['operator']);
                        break;
                    case 'condition_group':
                        $count->condition($filtro['condition']);
                        break;

                }
            }
        }
      */
    $total_rows = count($query->fetchAll(PDO::FETCH_ASSOC));
    $total_pages = ceil($total_rows / $options['limit']);

    if ($options['current_page']) {
      $current_page = $options['current_page'];
    } else {
      $current_page = 1;
    }


    /**
     *  Si la pagina actual es mayor que el total de paginas, lo iguala al total de paginas
     */
    if ($current_page > $total_pages) {
      $current_page = $total_pages;
    }


    /**
     *  Si la pagina actual es menor a 1, se iguala a 1
     */
    if ($current_page < 1) {
      $current_page = 1;
    }


    /**
     *  Sacar el offset para la consulta, si no se hace el -1, no mostraria los primeros 20,
     *  se resta la pagina actual (4 - 1 -> 3)* 20 -> 80 seria el offset que tomaria la consulta,
     *  pero en realidad ya esta en la cuarta pagina mostrando 100 items,
     *  por que los primeros 20 no los cuenta.
     */

    $offset = ($current_page - 1) * $options['limit'];


    /**
     * Render Pagination
     */
    $total_actual = $offset + $options['limit']; // se le aumenta el limite que serian los primeros que muestra la consulta y que no cuenta


    /**
     * si el total_actual se pasa por que se le suma el limite,
     * se hace un calculo para igualarlo, esto pasaria cuando se llegue al final de la paginación,
     * se calcula primero la diferencia que ahi entre el total de rows menos el total_actual,
     * despues se le resta el resultado de esa diferencia al limite,
     * para sacar la diferencia real sin la suma del limite,
     * despues se empareja el total a como estaria sin lo que se le sumo del limite,
     * que es el offset normal y por ultimo se suma el total sin limite ($total2)
     * mas la diferencia sin limite ($total3)
     */
    if ($total_actual >= $total_rows) {
      $total1 = (($total_actual - $total_rows));
      $total2 = $options['limit'] - $total1;
      $total3 = $total_actual - $options['limit'];
      $total_actual = $total3 + $total2;
    } else { // si no se le sigue sumando el limite
      $total_actual = $offset + $options['limit'];
    }


    /**
     * Inicio de Render de paginación
     */
    $paginacion = '<nav aria-label="Page navigation example"><ul class="pagination justify-content-center">';

    /**
     * Si es mayor que 1, para agregar las flechas para retroceder hasta la primera
     */
    if ($current_page > 1) {
      // retroceder hasta el final
      $paginacion .= " <li class='page-item'><a href='#' class='paginacion page-link' data-paginacion='1'><i class='fa fa-angle-double-left' aria-hidden='true'></i></a></li> ";
      $prevpage = $current_page - 1;
      // retroceder 1
      $paginacion .= " <li class='page-item'><a href='#' class='paginacion page-link' data-paginacion='$prevpage'><i class='fa fa-angle-left' aria-hidden='true'></i></a></li> ";
    }

    /**
     * Rango de items que saldran a cada lado , el actual va en el centro, y salen 3 para atras y 3 para delante
     */
    $range = $options['range'];


    for ($x = ($current_page - $range); $x < (($current_page + $range) + 1); $x++) {

      if (($x > 0) && ($x <= $total_pages)) {

        if ($x == $current_page) {
          // la pagina actual
          $paginacion .= " <li class='page-item active'><span class='page-link' >$x</span></li>";
        } else {
          $paginacion .= "<li class='page-item'><a href='#' class='paginacion page-link' data-paginacion='$x'>$x</a></li>  ";
        }
      }
    }


    /**
     * Agregar los de siguiente, y el de hasta el final
     */
    if ($current_page != $total_pages) {
      $nextpage = $current_page + 1;
      $paginacion .= " <li class='page-item'><a href='#' class='paginacion page-link' data-paginacion='$nextpage'><i class='fa fa-angle-right' aria-hidden='true'></i></a></li>  ";
      $paginacion .= " <li><a href='#' class='paginacion page-link' data-paginacion='$total_pages'><i class='fa fa-angle-double-right' aria-hidden='true'></i></a></li> ";
    }


    /**
     * Fin de Render de paginación
     */
    $paginacion .= '</ul></nav><p class="text-center">' . $total_actual . ' de ' . $total_rows . '</p>';


    return array($offset, $options['limit'], $paginacion);
  }

  /**
   * @param $nip {string} : El nip a buscar entre los usuarios.
   * @return bool: si no lográ encontrar un nip no regresa nada, de lo contrario regresa el nombre y usuario
   */
  public static function auth_user_nip($nip)
  {

    $get_all_passwords = db_select('partners_users_auth', 'pua');
    $get_all_passwords->leftJoin('users', 'u', 'u.uid = pua.uid');
    $get_all_passwords->fields('u', array('uid', 'name'));
    $get_all_passwords->fields('pua', array('nip'));
    $passwords = $get_all_passwords->execute()->fetchAll();
    $users_data = array();
    foreach ($passwords as $usr) {
      $users_data[] = array("nip" => $usr->nip, "uid" => $usr->uid, "name" => $usr->name);
    }

    $user_name = false;
    $user_id = false;
    foreach ($users_data as $user_pass) {
      $against = array("pass" => $user_pass['nip']);
      $against = (object) $against;
      if (user_check_password($nip, $against)) {
        $user_name = $user_pass['name'];
        $user_id = $user_pass['uid'];
      }
    }

    if (!$user_id) {
      return false;
    } else {
      return array($user_id, $user_name);
    }
  }

  public function lista_contactos($id_cliente, $estricto_facturacion)
  {
    $get_contactos = db_select('partners_clientes', 'pc');
    $get_contactos->fields('pc', array("id_cliente", "nombre_cliente", "alias", "nombre_facturacion", "pagina_web", "tipo_clasificacion", "credito", "tipo_cliente", "tipo_negocio"));
    $get_contactos->innerJoin('partners_clientes_datos', 'pcd', 'pc.id_cliente = pcd.id_cliente AND pcd.tipo_dato = 3');
    $get_contactos->addField('pcd', 'dato_valor', 'email');
    $get_contactos->condition('id_padre', $id_cliente);
    $get_contactos->condition('cancelado', 1, '!=');
    if ($estricto_facturacion == 1) {
      $get_contactos->condition('contacto_facturacion', 1);
    }
    $contactos = $get_contactos->execute()->fetchAll(PDO::FETCH_ASSOC);

    if (count($contactos) > 0) {
      echo json_encode(
        array(
          "status" => "success",
          "message" => json_encode($contactos)
        )
      );
    } else {
      echo json_encode(
        array(
          "status" => "error",
          "message" => "El cliente no tiene contactos, agrégalos."
        )
      );
    }
  }

  /**
   * Envia correos de una manera sencilla al pasar los parámetros necesarios.
   * @param $ruta
   * @param $ruta_data
   * @param $subject
   * @param $from
   * @param $to
   * @param null $attach
   * @param null $attach_title
   * @throws Exception
   */
  public function send_mail($ruta, $ruta_data, $subject, $from, $to, $attach = null, $attach_title = null)
  {
    $transport = new Swift_SendmailTransport();
    $mailer = new Swift_Mailer($transport);
    $body = render_template('php', $ruta, $ruta_data);
    $message = (new Swift_Message())
      ->setSubject($subject)
      ->setFrom($from)
      ->setBody($body, 'text/html');
    $failedRecipients = [];
    $numSent = 0;

    if ($attach != null) {
      $attachment = new Swift_Attachment($attach, $attach_title . '.pdf', 'application/pdf');
      $message->attach($attachment);
    }
    if (count($to) > 1) {
      foreach ($to as $address => $name) {
        if (is_int($address)) {
          $message->setTo($name);
        } else {
          $message->setTo([$address => $name]);
        }

        $numSent += $mailer->send($message, $failedRecipients);
        $this->register_log_actions($ruta, 'mail', 'Se envió un mensaje con subject: ' . $subject . ' a: ' . $name);
      }
    } else {
      $message->setTo($to);
      $numSent += $mailer->send($message, $failedRecipients);
      $this->register_log_actions($ruta, 'mail', 'Se envió un mensaje con subject: ' . $subject . ' a: ' . $name);

    }
  }

  /**
   * @param $html {string} a generar en pdf
   * @param $title {string} del pdf
   * @param $file_out_name {string} el nombre que quedará disponible al guardar el pdf
   * @param $type {string/int} 1: Vista normal, 2: Vista para email (solo string), 3: Guarda en servidor (POR EL MOMENTO SE DEJA LA RUTA ESPECFICICA)
   * @param $footer {string/int} Ruta del html del fotoer
   * @param $watermark {string} Ruta de la imagen del watermark
   * @return string
   * @throws MpdfException
   */
  protected function generate_pdf($html, $title, $file_out_name, $type, $footer = NULL, $watermark = NULL)
  {
    $mpdf = new mPDF();
    $mpdf->WriteHTML($html);
    $mpdf->SetTitle($title);
    if ($footer) {
      $mpdf->defaultfooterline = 0;
      $mpdf->SetHTMLFooter($footer);
    }
    if ($watermark) {
      $mpdf->SetWatermarkImage($watermark, 0.16, 'P', array(0, 5));
      $mpdf->showWatermarkImage = true;
    }
    if ($type == 1) {
      return $mpdf->Output($file_out_name . '.pdf', 'I');
    } else if ($type == 2) {
      return $mpdf->Output('', 'S');
    } else if ($type == 3) {
      return $mpdf->Output(HOME_SERVER . 'facturas_tmp/' . $file_out_name . '.pdf', 'F');
    }
  }

  protected function capturar_entrada_caja($concepto, $cantidad)
  {
    global $user;
    db_insert('partners_caja')
      ->fields(
        array(
          'concepto' => $concepto,
          'cantidad' => strtr($cantidad, array(',' => '')),
          'updated_at' => REQUEST_TIME,
          'created_at' => REQUEST_TIME,
          'created_by' => $user->uid,
          'updated_by' => $user->uid
        )
      )
      ->execute();

    $this->register_log_actions('partners_caja', 'insert', 'Se capturó en caja externamente bajo el concepto: ' . $concepto);
  }

  /**
   * @param $numero_factura {str} que quedará guardado
   * @param $id_registro {int} del registro procedente
   * @param $origen {str} origen de donde viene, para ayudar a rastrear el pago
   * @param $table_module {str} nombre de la tabla de origen, utilizado para saber el número de factura después
   * @param $folio {str} Folio para ingresar en las notas de la factura en cobranza.
   */
  protected function track_down_facturado($numero_factura, $id_registro, $origen, $table_module, $folio)
  {
    global $user;
    $track_number = db_insert('partners_seguimiento_pagos')
      ->fields(
        array(
          'created_at' => REQUEST_TIME,
          'created_by' => $user->uid,
          'id_registro' => $id_registro,
          'origen' => $origen,
          'numero_factura' => $numero_factura,
          'table_module' => $table_module
        )
      )
      ->execute();

    $check_exists = db_select('partners_cobranza', 'pc')
      ->fields('pc', array('folio'))
      ->condition('folio', $numero_factura)
      ->execute()->fetchField();

    if ($check_exists > 0) {
      $get_notas = db_select('partners_cobranza', 'pc');
      $get_notas->fields('pc', array('notas'));
      $get_notas->condition('folio', $numero_factura);
      $notas = $get_notas->execute()->fetchField();

      if ($notas != '' || $notas != NULL) {
        $new_notas = $notas . ',' . $folio;
      } else {
        $new_notas = $folio;
      }

      db_update('partners_cobranza')
        ->fields(
          array(
            'notas' => $new_notas
          )
        )
        ->condition('folio', $numero_factura)
        ->execute();
    } else {
      db_insert('partners_cobranza')
        ->fields(
          array(
            'folio' => $numero_factura,
            'apoyo' => 0,
            'notas' => $folio
          )
        )
        ->execute();
    }

    $this->register_log_actions('partners_seguimiento_pagos', 'insert', 'Se capturó un registro facturado para darle seguimiento a esa factura, id seguimiento: ' . $track_number);
  }

  /**
   * @param $table_module {str} tabla en la cual buscar el id_registro
   * @param $id_registro {str} a buscar el número de factura
   */
  protected function get_numero_factura($table_module, $id_registro)
  {
    $numero_factura = db_select('partners_seguimiento_pagos', 'psp')
      ->fields('psp', array('numero_factura'))
      ->condition('id_registro', $id_registro)
      ->condition('table_module', $table_module)
      ->orderBy('id_seguimiento', 'DESC')
      ->range(0, 1)
      ->execute()->fetchField();

    return $numero_factura;
  }

  protected function getToken($length)
  {
    $token = "";
    $codeAlphabet = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    $codeAlphabet .= "abcdefghijklmnopqrstuvwxyz";
    $codeAlphabet .= "0123456789";
    $max = strlen($codeAlphabet);
    for ($i = 0; $i < $length; $i++) {
      $token .= $codeAlphabet[self::crypto_rand_secure(0, $max)];
    }
    return $token;
  }

  protected function crypto_rand_secure($min, $max)
  {
    $range = $max - $min;
    if ($range < 1)
      return $min;
    $log = ceil(log($range, 2));
    $bytes = (int) ($log / 8) + 1;
    $bits = (int) $log + 1;
    $filter = (int) (1 << $bits) - 1;
    do {
      $rnd = hexdec(bin2hex(openssl_random_pseudo_bytes($bytes)));
      $rnd = $rnd & $filter;
    } while ($rnd >= $range);
    return $min + $rnd;
  }

  /**
   * Genera una key que será usada para encriptar y desencriptar.
   * @return Key
   * @throws \Defuse\Crypto\Exception\BadFormatException
   * @throws \Defuse\Crypto\Exception\EnvironmentIsBrokenException
   */
  protected function loadEncryptionKeyFromConfig()
  {
    $file = fopen(HOME_SERVER . 'etc/secret-key.txt', 'r');
    $keyAscii = fgets($file);
    fclose($file);
    return Key::loadFromAsciiSafeString($keyAscii);
  }
}