<?php
/**
 * Archivo del sistema interno de pcpartners.com.mx.
 * User: Erick
 * Date: 09/02/2019
 * Time: 12:51 PM
 */

namespace Partners\Vehiculo;


use Exception;
use Partners\Utils\GenerateTemplate;
use Partners\Utils\JsonMessageHelper;
use Partners\Utils\PartnersBitacora;
use Partners\Utils\PartnersException;
use Partners\Utils\PartnersLogger;
use PDO;

class VehiculoController
{

  /**
   * @var string
   */
  private $fileUploadFolder = HOME_SERVER . '/public_html/intranet/sites/default/files/vehiculos_archivos/';

  /**
   * La url la cual busca un archivo y lo muestra en pantalla.
   */
  private $mostrarArchivoUrl = '/vehiculos/consultas/mostrar_archivo';

  /**
   * @var int
   */
  private $fechaInicio = 0;

  /**
   * @var int
   */
  private $fechaFinal = 0;

  /**
   * @param $datos
   * @throws Exception
   */
  public function reporteBase($datos)
  {
    setlocale(LC_ALL, 'es_ES');
    $fi = str_replace('/', '-', trim($datos->filter->start_date));
    $ff = str_replace('/', '-', trim($datos->filter->end_date));
    $f_inicio = strtotime($fi);
    $f_final = strtotime($ff) + 86040;

    $this->fechaInicio = $f_inicio;
    $this->fechaFinal = $f_final;


    $query = db_select('partners_vehiculos', 'pv');
    $query->fields('pv');

    $query->orderBy('id_vehiculo', 'ASC');
    $query->condition("inactivo", "0");

    $vehiculos = [];
    $query_datos = $query->execute();
    while ($row = $query_datos->fetchAssoc()) {
      $Vehiculo = new Vehiculo($row);
      $Vehiculo->setGastoAnual($this->getGastoAnual($Vehiculo->getIdVehiculo()));
      $Vehiculo->setRefrendo($this->getLastRefrendo($Vehiculo->getIdVehiculo()));
      $Vehiculo->setSeguro($this->getLastSeguro($Vehiculo->getIdVehiculo()));
      $Vehiculo->setKilometraje($this->getLastKilometraje($Vehiculo->getIdVehiculo()));

      $nombreVehiculo = array("data" =>
        $Vehiculo->getNombreVehiculo() . ' ' . $Vehiculo->getAnoVehiculo(), "class" => array("cursor-pointer", "vista_vehiculo"));

      $placaVehiculo = array("data" => $Vehiculo->getPlacaVehiculo(), "class" => array("copy_to_clipboard", "cursor-pointer"));
      $serieVehiculo = array("data" => $Vehiculo->getSerieVehiculo(), "class" => array("copy_to_clipboard", "cursor-pointer"));

      $gastoAnual = array("data" =>
        number_format($Vehiculo->getGastoAnual(), 2, '.', ','),
        "class" => array("text-right", "cursor-pointer", "vista-gasto-anual"));

      $refrendo = array("data" => date('Y', $Vehiculo->getRefrendo()->getFechaRefrendo()),
        "class" => array("text-center", "cursor-pointer", "ver_refrendo")
      );

      $seguroVencimiento = strftime('%d-%b-%Y', $Vehiculo->getSeguro()->getFechaVencimiento());

      $seguroPrecio = number_format($Vehiculo->getSeguro()->getPrecio(), 2, '.', ',');
      $compania = $Vehiculo->getSeguro()->getCompania();
      $poliza = $Vehiculo->getSeguro()->getPoliza();

      $kilometraje = array("data" => number_format($Vehiculo->getKilometraje()->getKilometros(), 0),
        "class" => array("text-right", "cursor-pointer", "vista-kilometraje"));

      $seguroBlock = array("data" => "<table class='m-0 w-100'><td class='w-25 text-left'>$compania</td>
        <td class='w-25 text-left'>$poliza</td>
        <td class='w-25 text-center'>$seguroVencimiento</td>
        <td class='w-25 text-right'>$seguroPrecio</td></table>",
        "class" => array("text-center", "cursor-pointer", "ver_seguro", "p-0"));

      $vales = array("data" => "Ver", "class" => array("text-center", "cursor-pointer", "ver_vales"));

      $vehiculos[] = array("data" => array(
        $nombreVehiculo,
        $placaVehiculo,
        $serieVehiculo,
        $gastoAnual,
        $refrendo,
        $seguroBlock,
        $kilometraje,
        $vales
      ),
        "data-files-url" => $this->mostrarArchivoUrl,
        "data-id-vehiculo" => $Vehiculo->getIdVehiculo(),
        "data-seguro-file" => $Vehiculo->getSeguro()->getPdf(),
        "data-refrendo-file" => $Vehiculo->getRefrendo()->getPdf(),
        "data-nombre-vehiculo" => $Vehiculo->getNombreVehiculo()
      );
    }

    $header_cols = array(
      array("data" => "Vehículo"),
      array("data" => "Placa"),
      array("data" => "Serie"),
      array("data" => "Gasto Anual", "class" => "text-right"),
      array("data" => "Refrendo", "class" => "text-center"),
      array("data" => "<table class='m-0 w-100'><tr><td colspan='4' style='background: #fff;' class='text-center'> Seguro </td>
 <tr> <td class='w-25'>Compañia</td> <td class='w-25'>Poliza</td> <td class='w-25 text-center'>Vencimiento</td> <td class='w-25 text-right'>Precio</td> </tr> </tr></table>",
        "style" => "padding:0;"),
      array("data" => "Kilometraje", "class" => "text-right"),
      array("data" => "Vales"),
    );

    $table = array(
      "header" => $header_cols,
      "rows" => $vehiculos,
      "sticky" => false,
      "attributes" => array("id" => "table_reporteador_vehiculos", "class" => array("table-sm"))
    );

    $result = theme('table', $table);

    JsonMessageHelper::Success($result);
  }

  /**
   * Método para buscar el total del gasto anual de un vehículo.
   * El rango de fechas las toma de los atributos de la clase.
   * @param $idVehiculo
   * @return float
   */
  private function getGastoAnual($idVehiculo)
  {
    $getGasto = db_select('partners_vehiculos_gastos', 'pvg');
    $getGasto->addExpression('SUM(cantidad)', 'cantidad');
    $getGasto->condition('id_vehiculo', $idVehiculo);
    $getGasto->condition('fecha_gasto', array($this->fechaInicio, $this->fechaFinal), 'BETWEEN');
    $gasto = $getGasto->execute()->fetchField();

    return (float)$gasto;
  }

  /**
   * Método para traer el ultimo Refrendo registrado en el vehículo, no hace caso a las fechas
   * que tiene el reporteador en su estado actual.
   * @param $idVehiculo
   * @return Refrendo
   */
  private function getLastRefrendo($idVehiculo)
  {
    $getLastRefrendo = db_select('partners_vehiculos_refrendo', 'pvr');
    $getLastRefrendo->fields('pvr');
    $getLastRefrendo->condition('id_vehiculo', $idVehiculo);
    $getLastRefrendo->orderBy('id_refrendo', 'DESC');
    $getLastRefrendo->range(0, 1);
    $Refrendo = new Refrendo($getLastRefrendo->execute()->fetchAssoc());

    return $Refrendo;
  }

  /**
   * Método para buscar el último Seguro de un Vehiculo
   * @param $idVehiculo
   * @return Seguro
   */
  private function getLastSeguro($idVehiculo)
  {
    $getLastSeguro = db_select('partners_vehiculos_seguro', 'pvs');
    $getLastSeguro->fields('pvs');
    $getLastSeguro->condition('id_vehiculo', $idVehiculo);
    $getLastSeguro->orderBy('id_seguro', 'DESC');
    $getLastSeguro->range(0, 1);
    $Seguro = new Seguro($getLastSeguro->execute()->fetchAssoc());

    return $Seguro;
  }

  /**
   * Método para regresar el último kilometraje registrado de un Vehiculo.
   * @param $idVehiculo
   * @return Kilometraje
   */
  private function getLastKilometraje($idVehiculo)
  {
    $getLastKilometraje = db_select('partners_vehiculos_kilometraje', 'pvk');
    $getLastKilometraje->fields('pvk');
    $getLastKilometraje->condition('id_vehiculo', $idVehiculo);
    $getLastKilometraje->orderBy('id_kilometraje', 'DESC');
    $getLastKilometraje->range(0, 1);

    $lastKilometraje = $getLastKilometraje->execute()->fetchAssoc();
    if (!$lastKilometraje) {
      $Kilometraje = new Kilometraje([]);
    } else {
      $Kilometraje = new Kilometraje($getLastKilometraje->execute()->fetchAssoc());
    }

    return $Kilometraje;
  }

  /**
   * Método que llama a ambos métodos encargados de validar a ver si es necesario generar alertas ya sea por
   * seguro o afinación.
   */
  public function triggerAlerts()
  {
    $this->afinacionAlert();
    $this->seguroAlert();
  }

  /**
   * Método para generar las alertas(pendientes) de las afinaciones que lo necesiten.
   */
  private function afinacionAlert()
  {
    # Guardan los km que aguanta cada tipo de afinación
    $kmAfinacionNormal = 7000;
    $kmAfinacionAceite = 5000;

    # Por cada vehiculo traer su ultima afinacion
    /** @noinspection SqlNoDataSourceInspection */
    /** @noinspection SqlResolve */
    $dbQ = db_query("SELECT t.id_gasto,
       t.fecha_gasto,
       t.km,
       t.tipo_gasto,
       t.solo_aceite,
       t.id_vehiculo
        FROM (SELECT id_vehiculo, MAX(fecha_gasto) as max_fecha FROM partners_vehiculos_gastos b WHERE b.tipo_gasto = 2 group by b.id_vehiculo) m
       inner join partners_vehiculos_gastos t on t.id_vehiculo = m.id_vehiculo and t.fecha_gasto = m.max_fecha;
    ");

    while ($row = $dbQ->fetchAssoc()) {
      $idVehiculo = $row['id_vehiculo'];
      # Traer el vehiculo
      $Vehiculo = $this->findVehiculo($idVehiculo);

      # Traer el último Km del vehiculo.
      $Km = $this->getLastKilometraje($idVehiculo);

      $diffKm = $Km->getKilometros() - $Vehiculo->getUltimoKmAfinacion();

      // Si la ultima afinación fue de solo aceite, entonces la siguiente afinación es una afinación normal.
      if ($row['solo_aceite'] == 1) {
        if ($diffKm > $kmAfinacionAceite) {
          $desc = "La última afinación del vehículo " . $Vehiculo->getNombreVehiculo() .
            " fue solo aceite con " . $Vehiculo->getUltimoKmAfinacion() . " kilómetros, ahora " . $diffKm .
            " kilómetros despúes con " . $Km->getKilometros() . " km es necesario hacer una afinación completa.";

          $this->generarPendienteAlerta($desc, 'Próxima afinación necesaria');
        }
      } else if ($row['solo_aceite'] == 0) {
        // Si la ultima afinación fue normal, entonces la siguiente afinación es de solo aceite.
        if ($diffKm > $kmAfinacionNormal) {
          $desc = "La última afinación del vehículo " . $Vehiculo->getNombreVehiculo() .
            " fue afinación completa con " . $Vehiculo->getUltimoKmAfinacion() . " kilómetros, ahora " . $diffKm .
            " kilómetros despúes con " . $Km->getKilometros() . " km es necesario hacer un cambio solo aceite.";

          $this->generarPendienteAlerta($desc, 'Próxima afinación necesaria');
        }
      }
    }
  }

  /**
   * Método para buscar un vehículo en particular.
   * @param $idVehiculo
   * @return Vehiculo
   */
  private function findVehiculo($idVehiculo)
  {
    $getVehiculo = db_select('partners_vehiculos', 'pv');
    $getVehiculo->fields('pv');
    $getVehiculo->condition('id_vehiculo', $idVehiculo);
    $Vehiculo = new Vehiculo($getVehiculo->execute()->fetchAssoc());

    return $Vehiculo;
  }

  /**
   * Método para generar pendientes cuando exista una alerta de seguro o afinación.
   * @param $descripcionPendiente
   * @param $tituloPendiente
   */
  private function generarPendienteAlerta($descripcionPendiente, $tituloPendiente)
  {
    try {
      # A que usuarios se generará el pendiente.
      $users = [1, 8, 12]; // Roberto, Rocio

      foreach ($users as $user) {
        $idPendiente = db_insert('partners_pendientes')
          ->fields(array(
            'created_at' => REQUEST_TIME,
            'created_by' => $user,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $user,
            'tipo_pendiente' => 1,
            'area_pendiente' => 5, // Admon, ya que Roberto y Rocio están en esa área
            'asignado_para' => $user,
            'fecha_pendiente' => REQUEST_TIME,
            'descripcion_pendiente' => $descripcionPendiente,
            'prioridad_pendiente' => 1,
            'quien_reporta_pendiente' => 'Sistema automático de control de Vehículos.',
            'pendiente_otro' => $tituloPendiente
          ))
          ->execute();

        PartnersLogger::Info('partners_pendientes', 'insert', "Se generó una alerta de vehículos, idPendiente: $idPendiente");
      }
    } catch (Exception $exception) {
    }
  }

  /**
   * Método para generar las alertas (pendientes) de los seguros que ya están por vencer. (1 mes)
   */
  private function seguroAlert()
  {
    # Traer todos los seguros que no hayan sido enviado alertas (alerta_generada = 0) y que esten a menos de un mes
    $getSeguros = db_select('partners_vehiculos_seguro', 'pvs');
    $getSeguros->fields('pvs', array('id_seguro', 'id_vehiculo', 'fecha_vencimiento'));
    $getSeguros->condition('alerta_generada', 0);
    $getSeguros->condition('fecha_vencimiento', strtotime('now'), '>=');
    $getSeguros->condition('fecha_vencimiento', strtotime('next month'), '<');
    $seguros = $getSeguros->execute();

    while ($r = $seguros->fetchAssoc()) {
      # Generar pendiente
      $nombreVehiculo = $this->findVehiculo($r['id_vehiculo'])->getNombreVehiculo();
      $desc = "El seguro del vehículo " . $nombreVehiculo . " vencerá el día "
        . date('d-m-Y', $r['fecha_vencimiento']);
      $this->generarPendienteAlerta($desc, 'Seguro a vencer de ' . $nombreVehiculo);

      # Actualizar el seguro, marcar alerta_generada = 1
      db_update('partners_vehiculos_seguro')
        ->fields(array(
          'alerta_generada' => 1
        ))
        ->condition('id_seguro', $r['id_seguro'])
        ->execute();
    }
  }

  /**
   * @throws PartnersException
   */
  public function vistaNuevoVehiculo()
  {
    $html = GenerateTemplate::Generate('vehiculos.nuevo_vehiculo');
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para capturar un Vehículo en la base de datos.
   * @param $data
   */
  public function capturarVehiculo($data)
  {
    $Vehiculo = new Vehiculo((array)$data);
    if ($this->validarDatosObligatoriosVehiculo($Vehiculo)) {
      global $user;

      try {
        $idVehiculo = db_insert('partners_vehiculos')
          ->fields(array(
            'created_by' => $user->uid,
            'created_at' => REQUEST_TIME,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $user->uid,
            'nombre_vehiculo' => $Vehiculo->getNombreVehiculo(),
            'ano_vehiculo' => $Vehiculo->getAnoVehiculo(),
            'placa_vehiculo' => $Vehiculo->getPlacaVehiculo(),
            'serie_vehiculo' => $Vehiculo->getSerieVehiculo()
          ))
          ->execute();

        PartnersLogger::Info('paratners_vehiculos', 'insert', "Se capturó el vehículo $idVehiculo");
        JsonMessageHelper::Success('Bien, vehículo capturado correctamente!');
      } catch (Exception $ex) {
        PartnersLogger::Error($ex, 'capturarVehiculo');
      }
    } else {
      JsonMessageHelper::
      Error('Ocurrió un error al momento de validar los datos, comprueba que estés capturando todo correctamente.');
    }
  }

  /**
   * Método para validar que un objeto Vehiculo tenga los datos obligatorios
   * para poder capturarlo en la base de datos.
   * Regresa verdadero si los datos obligatorios están presentes y por lo tanto es posible
   * capturar el vehiculo.
   * Regresa falso cuando no se tienen todos los datos obligatorios.
   * @param Vehiculo $vehiculo
   * @return bool
   */
  private function validarDatosObligatoriosVehiculo(Vehiculo $vehiculo)
  {
    return (
      $vehiculo->getNombreVehiculo() != '' &&
      $vehiculo->getAnoVehiculo() != 0 &&
      $vehiculo->getPlacaVehiculo() != '' &&
      $vehiculo->getSerieVehiculo() != ''
    );
  }

  /**
   * @throws PartnersException
   */
  public function vistaNuevoGasto()
  {
    $listaVehiculos = $this->listaVehiculos();
    $html = GenerateTemplate::Generate('vehiculos.nuevo_gasto', $listaVehiculos);
    JsonMessageHelper::Success($html);
  }

  /**
   * Método que genera una lista de todos los vehículos existentes.
   */
  public function listaVehiculos()
  {
    $query = db_select('partners_vehiculos', 'pv');
    $query->fields('pv');
    /** @noinspection PhpUndefinedMethodInspection */
    $vehiculos = $query->execute()->fetchAll(PDO::FETCH_ASSOC);

    $listaVehiculos = [];
    foreach ($vehiculos as $vehiculo) {
      $Vehiculo = new Vehiculo($vehiculo);
      $listaVehiculos[] = $Vehiculo->__toArray();
    }

    return $listaVehiculos;
  }

  /**
   * Método para capturar un Gasto de un Vehiculo en la base de datos.
   * @param $data
   */
  public function capturarGasto($data)
  {
    $Gasto = new Gasto((array)$data);
    if ($this->validarDatosObligatoriosGasto($Gasto)) {
      global $user;

      try {
        $Bitacora = new PartnersBitacora($Gasto->getObservaciones());
        $idGasto = db_insert('partners_vehiculos_gastos')
          ->fields(array(
            'created_by' => $user->uid,
            'created_at' => REQUEST_TIME,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $user->uid,
            'id_vehiculo' => $Gasto->getIdVehiculo(),
            'concepto' => $Gasto->getConcepto(),
            'cantidad' => $Gasto->getCantidad(),
            'fecha_gasto' => $Gasto->getFechaGasto(),
            'taller' => $Gasto->getTaller(),
            'km' => $Gasto->getKm(),
            'observaciones' => $Bitacora->generarNuevaObservacion()
          ))
          ->execute();

        PartnersLogger::Info('paratners_vehiculos_gastos', 'insert', "Se capturó el gasto $idGasto");
        JsonMessageHelper::Success('Bien, Gasto capturado correctamente!');
      } catch (Exception $ex) {
        PartnersLogger::Error($ex, 'capturarGasto');
      }
    } else {
      JsonMessageHelper::
      Error('Ocurrió un error al momento de validar los datos, comprueba que estés capturando todo correctamente.');
    }
  }

  /**
   * Método usado para validar los datos obligatorios de un objeto de Gasto que se usaría
   * para capturarlo a la base de datos.
   * @param Gasto $gasto
   * @return bool
   */
  private function validarDatosObligatoriosGasto(Gasto $gasto)
  {
    return (
      $gasto->getIdVehiculo() != 0 &&
      $gasto->getConcepto() != '' &&
      $gasto->getFechaGasto() != '' &&
      $gasto->getCantidad() != 0 &&
      $gasto->getTaller() != '' &&
      $gasto->getKm() != 0
    );
  }

  /**
   * Método que genera una vista de un formulario para capturar un Seguro de un Vehículo.
   * @throws PartnersException
   */
  public function vistaNuevoSeguro()
  {
    $listaVehiculos = $this->listaVehiculos();
    $html = GenerateTemplate::Generate('vehiculos.nuevo_seguro', $listaVehiculos);
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para capturar un Seguro a un Vehículo
   * @param $data
   */
  public function capturarSeguro($data)
  {
    // Quitar comas del precio que en js no pude porque se construye la información con formData
    $data['data']['precio'] = str_replace(',', '', $data['data']['precio']);
    $Seguro = new Seguro($data['data']);
    if ($this->validarDatosObligatoriosSeguro($Seguro)) {
      global $user;

      // Meter imágen al servidor.
      $imgName = REQUEST_TIME . $data['files']['pdf']['name'];
      $imgTmpName = $data['files']['pdf']['tmp_name'];
      if (move_uploaded_file($imgTmpName, $this->fileUploadFolder . $imgName)) {
        $Seguro->setPdf($imgName);
      }

      try {
        $idSeguro = db_insert('partners_vehiculos_seguro')
          ->fields(array(
            'created_by' => $user->uid,
            'created_at' => REQUEST_TIME,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $user->uid,
            'id_vehiculo' => $Seguro->getIdVehiculo(),
            'fecha_vencimiento' => $Seguro->getFechaVencimiento(),
            'compania' => $Seguro->getCompania(),
            'poliza' => $Seguro->getPoliza(),
            'precio' => $Seguro->getPrecio(),
            'pdf' => $Seguro->getPdf(),
            'forma_pago' => $Seguro->getFormaPago()
          ))
          ->execute();

        PartnersLogger::Info('paratners_vehiculos_seguro', 'insert', "Se capturó el seguro $idSeguro");
        JsonMessageHelper::Success('Bien, Seguro capturado correctamente!');
      } catch (Exception $ex) {
        PartnersLogger::Error($ex, 'capturarSeguro');
      }
    } else {
      JsonMessageHelper::
      Error('Ocurrió un error al momento de validar los datos, comprueba que estés capturando todo correctamente.');
    }
  }

  /**
   * Método para validar los datos obligatorios de un objeto de Seguro que serían usados para capturarlo
   * en la base de datos.
   * @param Seguro $seguro
   * @return bool
   */
  private function validarDatosObligatoriosSeguro(Seguro $seguro)
  {
    return (
      $seguro->getIdVehiculo() != 0 &&
      $seguro->getFechaVencimiento() != 0 &&
      $seguro->getCompania() != '' &&
      $seguro->getPoliza() != '' &&
      $seguro->getPrecio() != 0 &&
      $seguro->getFormaPago() != ''
    );
  }

  /**
   * @throws PartnersException
   */
  public function vistaNuevoKm()
  {
    $html = GenerateTemplate::Generate('vehiculos.nuevo_km', $this->listaVehiculos());
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para capturar un kilometraje de un Vehiculo.
   * @param $data
   */
  public function capturarKm($data)
  {
    $Kilometraje = new Kilometraje((array)$data);
    if ($this->validarDatosObligatoriosKilometraje($Kilometraje)) {
      global $user;

      try {
        $idKilometraje = db_insert('partners_vehiculos_kilometraje')
          ->fields(array(
            'created_by' => $user->uid,
            'created_at' => REQUEST_TIME,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $user->uid,
            'id_vehiculo' => $Kilometraje->getIdVehiculo(),
            'kilometros' => $Kilometraje->getKilometros(),
          ))
          ->execute();

        # Aqui una vez capturado el kilometraje, se tiene que relacionar este kilometraje con el vale mediante id_control
        db_update('partners_vehiculos_control_vales')
          ->fields(array(
            'id_km' => $idKilometraje
          ))
          ->condition('id_control', $Kilometraje->getUnkownProperties()['id_control'])
          ->execute();

        PartnersLogger::Info('paratners_vehiculos_kilometraje', 'insert', "Se capturó el kilometraje $idKilometraje");
        JsonMessageHelper::Success('Bien, Kilometraje capturado correctamente!');
      } catch (Exception $ex) {
        PartnersLogger::Error($ex, 'capturarKm');
      }
    } else {
      JsonMessageHelper::
      Error('Ocurrió un error al momento de validar los datos, comprueba que estés capturando todo correctamente.');
    }
  }

  /**
   * Método para validar los datos obligatorios de un objeto de Kilometraje que serían usados para capturarlo
   * en la base de datos.
   * @param Kilometraje $kilometraje
   * @return bool
   */
  private function validarDatosObligatoriosKilometraje(Kilometraje $kilometraje)
  {
    return (
      $kilometraje->getIdVehiculo() != 0 &&
      $kilometraje->getKilometros() != 0
    );
  }

  /**
   * @throws PartnersException
   */
  public function vistaNuevaAfinacion()
  {
    $listaVehiculos = $this->listaVehiculos();
    $html = GenerateTemplate::Generate('vehiculos.nueva_afinacion', $listaVehiculos);
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para capturar una Afinación de un Vehiculo en la base de datos.
   * @param $data
   */
  public function capturarAfinacion($data)
  {
    $Gasto = new Gasto((array)$data);
    $Gasto->setTipoGasto(2);

    if ($this->validarDatosObligatoriosGasto($Gasto)) {
      global $user;

      try {
        $Bitacora = new PartnersBitacora($Gasto->getObservaciones());
        $idGasto = db_insert('partners_vehiculos_gastos')
          ->fields(array(
            'created_by' => $user->uid,
            'created_at' => REQUEST_TIME,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $user->uid,
            'id_vehiculo' => $Gasto->getIdVehiculo(),
            'concepto' => $Gasto->getConcepto(),
            'cantidad' => $Gasto->getCantidad(),
            'fecha_gasto' => $Gasto->getFechaGasto(),
            'taller' => $Gasto->getTaller(),
            'km' => $Gasto->getKm(),
            'observaciones' => $Bitacora->generarNuevaObservacion(),
            'tipo_gasto' => $Gasto->getTipoGasto(),
            'solo_aceite' => $Gasto->getSoloAceite(),
            //Campo de 'partners_vehiculos', no de 'partners_vehiculos_gastos
            //'ultimo_km_afinacion' => $Gasto->getKm() # Como es afinación, debe de actualizar este campo ya que guarda
            # el último km en el cual se hizó una afinación, será util para manejar las alertas después.
          ))
          ->execute();

        PartnersLogger::Info('paratners_vehiculos_gastos', 'insert', "Se capturó la afinación $idGasto");
        JsonMessageHelper::Success('Bien, Afinación capturada correctamente!');
      } catch (Exception $ex) {
        PartnersLogger::Error($ex, 'capturarAfinacion');
      }
    } else {
      JsonMessageHelper::
      Error('Ocurrió un error al momento de validar los datos, comprueba que estés capturando todo correctamente.');
    }
  }

  /**
   * Método que genera una vista de un formulario para capturar un Refrendo de un Vehículo.
   * @throws PartnersException
   */
  public function vistaNuevoRefrendo()
  {
    $listaVehiculos = $this->listaVehiculos();
    $html = GenerateTemplate::Generate('vehiculos.nuevo_refrendo', $listaVehiculos);
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para capturar un Refrendo a un Vehículo
   * @param $data
   */
  public function capturarRefrendo($data)
  {
    $Refrendo = new Refrendo($data['data']);
    if ($this->validarDatosObligatoriosRefrendo($Refrendo)) {
      global $user;

      // Meter imágen al servidor.
      $imgName = REQUEST_TIME . $data['files']['pdf']['name'];
      $imgTmpName = $data['files']['pdf']['tmp_name'];
      if (move_uploaded_file($imgTmpName, $this->fileUploadFolder . $imgName)) {
        $Refrendo->setPdf($imgName);
      }

      try {
        $idRefrendo = db_insert('partners_vehiculos_refrendo')
          ->fields(array(
            'created_by' => $user->uid,
            'created_at' => REQUEST_TIME,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $user->uid,
            'id_vehiculo' => $Refrendo->getIdVehiculo(),
            'pdf' => $Refrendo->getPdf(),
            'fecha_refrendo' => $Refrendo->getFechaRefrendo()
          ))
          ->execute();

        PartnersLogger::Info('paratners_vehiculos_refrendo', 'insert', "Se capturó el refrendo $idRefrendo");
        JsonMessageHelper::Success('Bien, Refrendo capturado correctamente!');
      } catch (Exception $ex) {
        PartnersLogger::Error($ex, 'capturarRefrendo');
      }
    } else {
      JsonMessageHelper::
      Error('Ocurrió un error al momento de validar los datos, comprueba que estés capturando todo correctamente.');
    }
  }

  /**
   * Método para validar los datos obligatorios de un objeto de Refrendo que serían usados para capturarlo
   * en la base de datos.
   * @param Refrendo $refrendo
   * @return bool
   */
  private function validarDatosObligatoriosRefrendo(Refrendo $refrendo)
  {
    return (
      $refrendo->getIdVehiculo() != 0 &&
      $refrendo->getFechaRefrendo() != 0
    );
  }

  /**
   * Método para generar la vista de un vehículo.
   * @param $idVehiculo
   * @throws PartnersException
   */
  public function vistaVehiculo($idVehiculo)
  {
    $Vehiculo = $this->findVehiculo($idVehiculo);
    $datos['vehiculo'] = $Vehiculo->__toArray();

    // Seguros
    $datos['seguro'] = $this->findSeguros($Vehiculo->getIdVehiculo());

    // Refrendo
    $datos['refrendo'] = $this->findRefrendos($Vehiculo->getIdVehiculo());

    // URL
    $datos['url_files'] = $this->mostrarArchivoUrl;

    $html = GenerateTemplate::Generate('vehiculos.vista_vehiculo', $datos);
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para buscar todos los seguros de un vehículo en particular
   * @param $idVehiculo
   * @return array
   */
  private function findSeguros($idVehiculo)
  {
    $getSeguros = db_select('partners_vehiculos_seguro', 'pvs');
    $getSeguros->fields('pvs');
    $getSeguros->condition('id_vehiculo', $idVehiculo);
    $seguros = $getSeguros->execute();

    $listSeguros = [];
    while ($row = $seguros->fetchAssoc()) {
      $row['url_files'] = $this->mostrarArchivoUrl;
      $seguro = new Seguro($row);
      $listSeguros[] = $seguro->__toArray();
    }

    return $listSeguros;
  }

  /**
   * Método para consultar todos los refrendos de un vehiculo.
   * @param $idVehiculo
   * @return array
   */
  private function findRefrendos($idVehiculo)
  {
    $getRefrendos = db_select('partners_vehiculos_refrendo', 'pvr');
    $getRefrendos->fields('pvr');
    $getRefrendos->condition('id_vehiculo', $idVehiculo);
    $refrendos = $getRefrendos->execute();

    $listaRefrendos = [];
    while ($row = $refrendos->fetchAssoc()) {
      $row['url_files'] = $this->mostrarArchivoUrl;
      $refrendo = new Refrendo($row);
      $listaRefrendos[] = $refrendo->__toArray();
    }

    return $listaRefrendos;
  }

  /**
   * @throws PartnersException
   */
  public function vistaKilometraje()
  {
    $html = GenerateTemplate::Generate('vehiculos.vista_kilometraje');
    JsonMessageHelper::Success($html);
  }

  /**
   * @param $datos
   * @throws Exception
   */
  public function reporteKilometraje($datos)
  {
    $kilometraje = $this->findKilometraje($datos->idVehiculo, strtotime($datos->start), strtotime($datos->end));

    $kmRecorridos = 0;
    if (count($kilometraje) > 0) {
      $kmRecorridos = abs($kilometraje[0]['kilometros'] - $kilometraje[count($kilometraje) - 1]['kilometros']);
    }
    $kilometrajes = [];
    foreach ($kilometraje as $km) {
      $kilometrajes[] = array(
        "data" => array(
          date('d-m-Y', $km['createdAt']),
          number_format($km['kilometros'], 0, '.', ',')
        )
      );
    }

    $header_cols = array(
      array("data" => "Fecha"),
      array("data" => "Kilómetros")
    );

    $table = array(
      "header" => $header_cols,
      "rows" => $kilometrajes,
      "sticky" => false,
      "attributes" => array("id" => "table_reporteador_kilometrajes", "class" => array("table-sm"))
    );

    $result = theme('table', $table);

    JsonMessageHelper::Success(["table" => $result, "km_recorridos" => $kmRecorridos]);
  }

  /**
   * Método que buscar todos los kilometrajes registrados de un vehiculo.
   * Tiene la posibilidad de buscar dentro de un rango de fechas solamente al pasarle
   * los parametros necesarios.
   * @param $idVehiculo
   * @param bool $startDate
   * @param bool $endDate
   * @return array
   */
  private function findKilometraje($idVehiculo, $startDate = false, $endDate = false)
  {
    $getKilometrajes = db_select('partners_vehiculos_kilometraje', 'pvk');
    $getKilometrajes->fields('pvk');
    $getKilometrajes->condition('id_vehiculo', $idVehiculo);
    if ($startDate && $endDate) {
      $getKilometrajes->condition('created_at', array($startDate, $endDate), 'BETWEEN');
    }

    $kilometros = $getKilometrajes->execute();

    $listaKilometraje = [];
    while ($row = $kilometros->fetchAssoc()) {
      $Kilometraje = new Kilometraje($row);
      $listaKilometraje[] = $Kilometraje->__toArray();
    }

    return $listaKilometraje;
  }

  /**
   * @throws PartnersException
   */
  public function vistaGastoAnual()
  {
    $html = GenerateTemplate::Generate('vehiculos.vista_gasto_anual');
    JsonMessageHelper::Success($html);
  }

  /**
   * @param $datos
   * @throws Exception
   */
  public function reporteGastoAnual($datos)
  {
    $gastoAnual = $this->findGastoAnual($datos->idVehiculo, strtotime($datos->start), strtotime($datos->end));

    $gastosAnuales = [];
    $totalGasto = 0;
    foreach ($gastoAnual as $gasto) {
      $totalGasto += $gasto['cantidad'];

      # Observaciones
      if ($gasto['observaciones']) {
        $obs = array("data" => "Obs", "class" => array("status-yellow", "text-center", "cursor-pointer", "status_obs"));
      } else {
        $obs = array("data" => "Obs", "class" => array("text-center", "cursor-pointer", "status_obs"));
      }
      $gastosAnuales[] = array(
        "data" => array(
          date('d-m-Y', $gasto['fechaGasto']),
          $gasto['concepto'],
          $gasto['taller'],
          array("data" => number_format($gasto['km'], 0, '.', ','), "class" => array("text-right")),
          array("data" => number_format($gasto['cantidad'], 2, '.', ','), "class" => array("text-right")),
          $obs
        ),
        "data-id-gasto" => $gasto['idGasto']
      );
    }

    // Agregar el tr del total
    $gastosAnuales[] = array(
      "data" => array(
        array("data" => "<b>Gasto total: </b>", "colspan" => 4, "class" => array("text-right")),
        array("data" => "<b>" . number_format($totalGasto, 2, '.', ',') . "</b>",
          "class" => array("text-right")),
      ),
      "colspan" => "4"
    );

    $header_cols = array(
      array("data" => "Fecha"),
      array("data" => "Concepto"),
      array("data" => "Taller"),
      array("data" => "Km", "class" => "text-right"),
      array("data" => "Cantidad", "class" => "text-right"),
      array("data" => "Obs", "class" => "text-center"),
    );

    $table = array(
      "header" => $header_cols,
      "rows" => $gastosAnuales,
      "sticky" => false,
      "attributes" => array("id" => "table_reporteador_gasto_anual", "class" => array("table-sm"))
    );

    $result = theme('table', $table);

    JsonMessageHelper::Success($result);
  }

  /**
   * Método que buscar todos los gastos registrados de un vehiculo.
   * Tiene la posibilidad de buscar dentro de un rango de fechas solamente al pasarle
   * los parametros necesarios.
   * @param $idVehiculo
   * @param bool $startDate
   * @param bool $endDate
   * @return array
   */
  private function findGastoAnual($idVehiculo, $startDate = false, $endDate = false)
  {
    $getGastoAnual = db_select('partners_vehiculos_gastos', 'pvg');
    $getGastoAnual->fields('pvg');
    $getGastoAnual->condition('id_vehiculo', $idVehiculo);
    if ($startDate && $endDate) {
      $getGastoAnual->condition('fecha_gasto', array($startDate, $endDate), 'BETWEEN');
      $getGastoAnual->orderBy('fecha_gasto', 'ASC');
    }

    $gastosAnuales = $getGastoAnual->execute();

    $listaGastos = [];
    while ($row = $gastosAnuales->fetchAssoc()) {
      $Gasto = new Gasto($row);
      $listaGastos[] = $Gasto->__toArray();
    }

    return $listaGastos;
  }

  /**
   * Método para controlar las observaciones de los gastos.
   * @param $datos
   * @throws Exception
   */
  public function loadObs($datos)
  {
    $Bitacora = new PartnersBitacora();
    $Bitacora->setTableName($datos->tabla);
    $Bitacora->setColName($datos->col);
    $Bitacora->setId($datos->idIndex);
    $Bitacora->setPrimaryKey($datos->index);

    // Consultar las observaciones
    if ($datos->type === 1) {
      $obs = $Bitacora->loadObs();
      JsonMessageHelper::Success($obs);
    } else if ($datos->type === 2) {
      // Capturar una observacion
      $Bitacora->setTextoObservacion($datos->new_val);
      $newObs = $Bitacora->capturarObs();

      JsonMessageHelper::Success($newObs);
    }
  }

  /**
   * Método para generar una vista del formulario para capturar entrada de vales
   * @throws PartnersException
   */
  public function vistaNuevaEntradaVales()
  {
    $listaVehiculos = $this->listaVehiculos();
    $html = GenerateTemplate::Generate('vehiculos.nueva_entrada_vales', $listaVehiculos);
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para capturar una entrada de vales.
   * @param $datos
   */
  public function capturarEntradaVales($datos)
  {
    $EntradaVales = new EntradaVales((array)$datos);
    # Armar array de lista Vales
    $listVales = [];
    foreach ($datos->listVales as $listVale) {
      $listVales[] = (array)$listVale;
    }
    $EntradaVales->setListVales($listVales);

    if ($this->validarDatosObligatoriosEntradaVales($EntradaVales)) {
      global $user;

      try {
        $idEntradaVales = db_insert('partners_vehiculos_entrada_vales')
          ->fields(array(
            'created_at' => REQUEST_TIME,
            'created_by' => $user->uid,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $user->uid,
            'precio_gasolina' => $EntradaVales->getPrecioGasolina()
          ))
          ->execute();

        foreach ($EntradaVales->getListVales() as $vale) {
          db_insert('partners_vehiculos_entrada_vales_datos')
            ->fields(array(
              'id_entrada_vales' => $idEntradaVales,
              'folio_vale' => $vale['valeFolio'],
              'monto_vale' => $vale['valeMonto'],
              'litros' => ($vale['valeMonto'] / $EntradaVales->getPrecioGasolina())
            ))
            ->execute();
        }
        PartnersLogger::Info('paratners_vehiculos_entrada_vales', 'insert', "Se capturó la entrada de vales $idEntradaVales");
        JsonMessageHelper::Success('Bien, Entrada de vales capturada correctamente!');
      } catch (Exception $ex) {
        PartnersLogger::Error($ex, 'capturarEntradaVales');
      }
    } else {
      JsonMessageHelper::
      Error('Ocurrió un error al momento de validar los datos, comprueba que estés capturando todo correctamente.');
    }
  }

  /**
   * @param EntradaVales $entradaVales
   * @return bool
   */
  public function validarDatosObligatoriosEntradaVales(EntradaVales $entradaVales)
  {
    return (count($entradaVales->getListVales()) > 0);
  }

  /**
   * Método para generar una vista del formulario para capturar un vale
   * @throws PartnersException
   */
  public function vistaNuevoVale()
  {
    $listaVehiculos = $this->listaVehiculos();
    $listaUsuarios = $this->listaUsuarios();
    $listaVales = $this->listaValesActivos();

    $datos['usuarios'] = $listaUsuarios;
    $datos['vehiculos'] = $listaVehiculos;
    $datos['vales'] = $listaVales;
    $html = GenerateTemplate::Generate('vehiculos.nuevo_vale', $datos);
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para retornar una lista de usuarios activos del sistema.
   * @return array
   */
  public function listaUsuarios()
  {
    $getUsers = db_select('users', 'u');
    $getUsers->fields('u', array('uid', 'name'));
    $getUsers->condition('status', 1);
    $getUsers->condition('uid', [37, 89, 26], 'NOT IN'); // Veronica, Soporte Recepción, Admin
    /** @noinspection PhpUndefinedMethodInspection */
    $users = $getUsers->execute()->fetchAll(PDO::FETCH_ASSOC);

    return $users;
  }

  /**
   * Método que regresa una lista de todos los vales que siguen activos para ser utilizados.
   * @return array
   */
  public function listaValesActivos()
  {
    $getListVales = db_select('partners_vehiculos_entrada_vales_datos', 'pvevd');
    $getListVales->fields('pvevd');
    $getListVales->condition('vale_usado', 0);
    /** @noinspection PhpUndefinedMethodInspection */
    $listaVales = $getListVales->execute()->fetchAll(PDO::FETCH_ASSOC);

    return $listaVales;
  }

  /**
   * Método para capturar un vale
   * @param $datos
   */
  public function capturarVale($datos)
  {
    try {
      $idControlVale = db_insert('partners_vehiculos_control_vales')
        ->fields(array(
          'id_vale' => $datos->id_vale,
          'id_vehiculo' => $datos->id_vehiculo,
          'usuario_recibe' => $datos->usuario_recibe,
          'fecha_usado' => $datos->fecha_usado
        ))
        ->execute();

      # Marcar el vale como usado
      db_update('partners_vehiculos_entrada_vales_datos')
        ->fields(array(
          'vale_usado' => 1
        ))
        ->condition('id_vale', $datos->id_vale)
        ->execute();

      PartnersLogger::Info('paratners_vehiculos_control_vales', 'insert', "Se capturó un vale entregado $idControlVale");
      JsonMessageHelper::Success('Bien, Vale entregado correctamente!');
    } catch (Exception $ex) {
      PartnersLogger::Error($ex, 'capturarEntradaVales');
    }
  }

  /**
   * Método para generar una vista de vales de un vehículo.
   * @throws PartnersException
   */
  public function vistaVales()
  {
    $html = GenerateTemplate::Generate('vehiculos.vista_vales');
    JsonMessageHelper::Success($html);
  }

  /**
   * Método que genera un reporte de los vales de un vehículo en especifico dentro de unas fechas en especifico.
   * @param $datos
   * @throws Exception
   */
  public function reporteVales($datos)
  {
    setlocale(LC_ALL, 'es_ES');
    $getVales = db_select('partners_vehiculos_control_vales', 'pvcv');
    $getVales->fields('pvcv');
    $getVales->condition('id_vehiculo', $datos->idVehiculo);
    $getVales->condition('fecha_usado', array(strtotime($datos->start), strtotime($datos->end)), 'BETWEEN');
    $getVales->orderBy('fecha_usado', 'ASC');
    $vales = $getVales->execute();

    $listVales = [];
    $totalMonto = 0;
    $totalKmRecorridos = 0;
    $totalKmxL = 0;
    while ($row = $vales->fetchAssoc()) {
      # Por cada vale que encuentre, se tiene que traer lo siguiente:
      # * El vale siguiente en caso de que exista para:
      #   * Sacar los kilometros que recorrio el vale actual
      #   * Sacar el Rendimiento/litro que recorrio el vale actual
      # * Los kilometros que tenia el carro en la fecha que se entregó el vale.

      # Traer el km del siguiente vale
      $nextValeKm = $this->getClosestValeKm($row['id_vehiculo'], $row['fecha_usado']);

      # Current km, diff km
      $Km = $this->getKilometraje($row['id_km']);
      if ($nextValeKm == 0) {
        $diffKm = 0;
      } else {
        $diffKm = abs($Km->getKilometros() - $nextValeKm);
      }

      # Current Vale
      $Vale = $this->findVale($row['id_vale']);

      #Km/l
      $kmxLitro = ($diffKm / $Vale->getLitros());

      $listVales[] = array(
        "data" => array(
          $Vale->getFolioVale(),
          strftime('%d-%b', $row['fecha_usado']),
          array("data" => number_format($Vale->getMontoVale(), 2, '.', ','), "class" => array("text-right")),
          array("data" => number_format($Vale->getLitros(), 0, '.', ','), "class" => array("text-right")),
          array("data" => number_format($Km->getKilometros(), 0, '.', ','), "class" => array("text-right")),
          array("data" => number_format($diffKm, 0, '.', ','), "class" => array("text-right")),
          array("data" => number_format($kmxLitro, 2, '.', ','), "class" => array("text-right"))
        )
      );

      $totalKmRecorridos += $diffKm;
      $totalMonto += $Vale->getMontoVale();
      $totalKmxL += $kmxLitro;
    }

    $listVales[] = array(
      "data" => array(
        array("data" => number_format($totalMonto, 2, '.', ','),
          "class" => array("text-right", "font-weight-bold"), "colspan" => 3),
        array("data" => number_format($totalKmRecorridos, 0, '.', ','),
          "class" => array("text-right", "font-weight-bold"), "colspan" => 3),
        array("data" => number_format(($totalKmxL / count($listVales)), 2, '.', ','),
          "class" => array("text-right", "font-weight-bold")),
      )
    );

    $header_cols = array(
      array("data" => "Folio"),
      array("data" => "Fecha"),
      array("data" => "Monto", "class" => array("text-right")),
      array("data" => "Litros", "class" => array("text-right")),
      array("data" => "Último Km", "class" => array("text-right")),
      array("data" => "Km Recorridos", "class" => array("text-right")),
      array("data" => "Km/l", "class" => array("text-right")),
    );

    $table = array(
      "header" => $header_cols,
      "rows" => $listVales,
      "sticky" => false,
      "attributes" => array("id" => "table_reporteador_vales", "class" => array("table-sm"))
    );

    $result = theme('table', $table);

    JsonMessageHelper::Success($result);
  }

  /**
   * Método para encontrar el vale siguiente más cercano de un vale en especifico.
   * @param $idVehiculo
   * @param $date
   * @return int
   */
  private function getClosestValeKm($idVehiculo, $date)
  {
    $getClosestVale = db_select('partners_vehiculos_control_vales', 'pvcv');
    $getClosestVale->fields('pvcv', array('id_km'));
    $getClosestVale->where("fecha_usado = (SELECT MIN(aux.fecha_usado) FROM partners_vehiculos_control_vales aux where aux.fecha_usado > $date and aux.id_vehiculo = $idVehiculo)");
    $getClosestVale->condition('id_vehiculo', $idVehiculo);
    $closestVale = $getClosestVale->execute()->fetchAssoc();
    if (!$closestVale) {
      $valeKm = 0;
    } else {
      $getKmVale = db_select('partners_vehiculos_kilometraje', 'pvk');
      $getKmVale->addField('pvk', 'kilometros');
      $getKmVale->condition('id_kilometraje', $closestVale);
      $valeKm = $getKmVale->execute()->fetchField();
    }

    return $valeKm;
  }

  /**
   * Método para regresar un kilometraje en especifico.
   * @param $idKm
   * @return Kilometraje
   */
  private function getKilometraje($idKm)
  {
    $getKm = db_select('partners_vehiculos_kilometraje', 'pvk');
    $getKm->fields('pvk');
    $getKm->condition('id_kilometraje', $idKm);
    $km = $getKm->execute()->fetchAssoc();

    return new Kilometraje($km);
  }

  /**
   * Método para regresar un Vale en especifico.
   * @param $idVale
   * @return Vale
   */
  private function findVale($idVale)
  {
    $getVale = db_select('partners_vehiculos_entrada_vales_datos', 'pvevd');
    $getVale->fields('pvevd');
    $getVale->condition('id_vale', $idVale);
    $vale = $getVale->execute()->fetchAssoc();

    return new Vale($vale);
  }

  /**
   * Método para generar un formulario para capturar un kilometraje de un vehiculo.
   * @return string
   */
  public function vistaNuevoKmMobile()
  {
    $html = "";
    try {
      $html = GenerateTemplate::Generate('vehiculos.nuevo_km_mobile', $this->listaVehiculos());
    } catch (PartnersException $e) {
    }
    return $html;
  }

  /**
   * Método para generar una lista de todos los vales de un vehículo que no han sido relacionados con un kilometraje.
   * @param $datos
   */
  public function listValesVehiculo($datos)
  {
    $idVehiculo = $datos['id_vehiculo'];
    $getVales = db_select('partners_vehiculos_control_vales', 'pvcv');
    $getVales->innerJoin('partners_vehiculos_entrada_vales_datos', 'pvevd', 'pvcv.id_vale = pvevd.id_vale');
    $getVales->fields('pvevd', array('folio_vale', 'id_vale', 'monto_vale'));
    $getVales->fields('pvcv', array('id_control'));
    $getVales->condition('pvcv.id_vehiculo', $idVehiculo);
    $getVales->condition('pvcv.id_km', NULL, 'IS');
    /** @noinspection PhpUndefinedMethodInspection */
    $vales = $getVales->execute()->fetchAll(PDO::FETCH_ASSOC);
    JsonMessageHelper::Success($vales);
  }

  /**
   * Método para mostrar un archivo. Se hace de esta manera para que al usuario nunca se le muestre la ruta
   * en la que está guardado el archivo.
   * @param $filename
   */
  public function showPdfFile($filename)
  {
    $remoteImage = $this->fileUploadFolder . $filename['file'];
    header("Content-type: application/pdf");
    readfile($remoteImage);
  }

  /**
   * Método para generar un reporte de todos los vales existentes.
   * @throws Exception
   */
  public function vistaReporteVales(){
    $db = db_select('partners_vehiculos_entrada_vales_datos', 'pvevd');
    $db->leftJoin('partners_vehiculos_control_vales', 'pvcv', 'pvevd.id_vale = pvcv.id_vale');
    $db->leftJoin('partners_vehiculos', 'pv', 'pvcv.id_vehiculo = pv.id_vehiculo');
    $db->fields('pvevd', array(
      'folio_vale',
      'monto_vale',
      'litros',
      'vale_usado'
    ));
    $db->addField('pv', 'nombre_vehiculo');
    $db->fields('pvcv', array('fecha_usado', 'usuario_recibe'));
    $db->orderBy('fecha_usado');
    $query = $db->execute();

    $vales = [];
    while($row = $query->fetchAssoc()){
      if($row['vale_usado']){
        $fecha = array("data" => date('d/m/Y', $row['fecha_usado']), "class" => array("text-center"));
        $nombreVehiculo = $row['nombre_vehiculo'];
        $usuario = user_load($row['usuario_recibe'])->name;
      }else{
        $fecha = array('data' => 'Sin Usar', 'class' => 'text-center');
        $nombreVehiculo = 'N/A';
        $usuario = 'N/A';
      }

      $vales[] = array(
        "data" => array(
          $row['folio_vale'],
          array("data" => number_format($row['monto_vale'], 2, '.', ','), "class" => array("text-right")),
          array("data" => number_format($row['litros'], 0, '.', ','), "class" => array("text-right")),
          $fecha,
          $nombreVehiculo,
          $usuario
        )
      );
    }

    $header_cols = array(
      array("data" => "Folio", "class" => "filter-number"),
      array("data" => "Monto", "class" => array("text-right", "filter-number")),
      array("data" => "Litros", "class" => array("text-right", "filter-number")),
      array("data" => "Fecha Usado", "class" => array("text-center", "filter-date")),
      array("data" => "Vehículo", "class" => "filter-text"),
      array("data" => "Usuario", "class" => "filter-text"),
    );
    $table = array(
      "header" => $header_cols,
      "rows" => $vales,
      "sticky" => false,
      "attributes" => array("id" => "table_reporte_vales", "class" => array("table-sm"))
    );

    $result = theme('table', $table);

    JsonMessageHelper::Success($result);
  }
}
