<?php
/**
 * Created by PhpStorm.
 * User: Erick
 * Date: 01/02/2019
 * Time: 06:04 PM
 */

namespace Partners\Cliente;


use Partners\Cotizacion\Cotizacion;
use Partners\Renovacion\Renovacion;
use Partners\Utils\GenerateTemplate;
use PDO;
use sys_tools;

/**
 * Class ClienteController
 * @package Partners\Cliente
 */
class ClienteController
{
  /**
   * Método para buscar a un Cliente en particular en base a su ID.
   * @param $idCliente
   * @return Cliente
   */
  public function findCliente($idCliente)
  {
    $db = db_select('partners_clientes', 'pc');
    $db->fields('pc');
    $db->addExpression("CASE pc.tipo_cliente WHEN 1 THEN 'Cliente' WHEN 2 THEN 'Contacto' WHEN 3 THEN 'Proveedor' END ", 'tipo_cliente');
    $db->addExpression("CASE pc.tipo_negocio WHEN 1 THEN 'Negocio' WHEN 2 THEN 'Particular' WHEN 3 THEN 'Distribuidor' WHEN 4 THEN 'Prospecto' WHEN 5 THEN 'Proveedor' END ", 'negocio');
    $db->condition('id_cliente', $idCliente);

    $Cliente = new Cliente($db->execute()->fetchAssoc());

    # Traer sus datos
    $Cliente->setDatosCliente($this->getOnlyDatosCliente($idCliente));

    # Traer sus datos de facturación
    $Cliente->setDatosFacturacion($this->getOnlyDatosFacturacion($idCliente));

    # Traer sus licencias
    $Cliente->setClienteLicencias($this->getOnlyClienteLicencias($idCliente));
    return $Cliente;
  }

  /**
   * @param $idCliente
   * @return array
   */
  private function getOnlyDatosCliente($idCliente)
  {
    $datosCliente = [];
    $queryD = db_select('partners_clientes_datos', 'pcd');
    $queryD->fields('pcd');
    $queryD->condition('id_cliente', $idCliente);
    $queryD->orderBy('FIELD(pcd.tipo_dato,1,11,2,12,3,13,5,6,7,8,9,10)');
    /** @noinspection PhpUndefinedMethodInspection */
    $clienteDatos = $queryD->execute()->fetchAll(PDO::FETCH_ASSOC);
    foreach ($clienteDatos as $clienteDato) {
      $DatosCliente = new DatosCliente($clienteDato);
      $datosCliente[] = $DatosCliente->__toArray();
    }

    return $datosCliente;
  }

  /**
   * @param $idCliente
   * @return array
   */
  private function getOnlyDatosFacturacion($idCliente)
  {
    $datos = [];
    $queryDatosFacturacion = db_select('partners_clientes_datos_facturacion', 'pcdf');
    $queryDatosFacturacion->fields('pcdf');
    $queryDatosFacturacion->condition('id_cliente', $idCliente);
    /** @noinspection PhpUndefinedMethodInspection */
    $datosFacturacion = $queryDatosFacturacion->execute()->fetchAll(PDO::FETCH_ASSOC);

    foreach ($datosFacturacion as $datoFacturacion) {
      $DatosFacturacion = new DatosFacturacion($datoFacturacion);
      $datos[] = $DatosFacturacion->__toArray();
    }

    return $datos;
  }

  /**
   * @param $idCliente
   * @return array
   */
  private function getOnlyClienteLicencias($idCliente)
  {
    $licencias = [];
    $getLic = db_select('partners_clientes_licencias', 'pclt');
    $getLic->fields('pclt');
    $getLic->condition('id_cliente', $idCliente);
    $getLic->orderBy('id_licencia', 'DESC');
    /** @noinspection PhpUndefinedMethodInspection */
    $datosLicencias = $getLic->execute()->fetchAll(PDO::FETCH_ASSOC);

    foreach ($datosLicencias as $datosLicencia) {
      $DatosLicencia = new ClienteLicencia($datosLicencia);
      $licencias[] = $DatosLicencia->__toArray();
    }

    return $licencias;
  }

  /**
   * Método para buscar todas las cotizaciones relacionadas a un solo Cliente.
   * @param $idCliente
   * @return array
   */
  public function findCotizacionesCliente($idCliente)
  {
    $query = db_select('partners_cotizaciones', 'pc');
    $query->fields('pc');
    $query->leftJoin('partners_cotizaciones_datos', 'pcd', 'pcd.id_cotizacion = pc.id_cotizacion');
    $query->addExpression('SUM(pcd.cantidad * pcd.precio)', 'subtotal');
    $query->addExpression("GROUP_CONCAT(pcd.descripcion SEPARATOR '|')", 'desc_cotizacion');
    $query->condition('pc.cotizacion_eliminada', 1, '!=');
    $query->condition('pc.id_cliente', $idCliente);
    $query->orderBy('id_cotizacion', 'DESC');
    $query->groupBy('pc.id_cotizacion');

    $get_cot = $query->execute();

    $cotizaciones = [];
    while ($row = $get_cot->fetchAssoc()) {
      $cotizacion = new Cotizacion($row);
      $cotizaciones[] = $cotizacion->__toArray();
    }

    return $cotizaciones;
  }

  /**
   * @param $idCliente
   * @return array
   */
  public function findRenovacionesCliente($idCliente)
  {
    $query = db_select('partners_renovaciones', 'pr');
    $query->innerJoin('taxonomy_term_data', 'pt', 'pt.tid = pr.id_tipo');
    $query->fields('pr');
    $query->addField('pt', 'name', 'nombre_tipo_renovacion');
    $query->condition('pr.renovacion_eliminada', '1', '!=');
    $query->condition('pr.fecha_renovacion', array(strtotime('first day of january this year'), (strtotime('last day of december this year') + 86340)), 'BETWEEN');
    $query->condition('pr.id_cliente', $idCliente);
    $query->orderBy('fecha_renovacion', 'DESC');
    $query->condition('pr.renovacion_eliminada', '1', '!=');

    $get_ren = $query->execute();

    $renovaciones = [];
    while ($row = $get_ren->fetchAssoc()) {
      $Renovacion = new Renovacion($row);
      $renovaciones[] = $Renovacion->__toArray();
    }

    return $renovaciones;
  }

  /**
   * Método para generar la vista de ventas de un cliente en especifico.
   * @param $idCliente
   * @return string
   * @throws \Exception
   */
  public function generarVistaVentas($idCliente)
  {
    $Cliente = $this->findCliente($idCliente);
    $cotizacionesCliente = $this->findCotizacionesCliente($idCliente);
    $renovacionesCliente = $this->findRenovacionesCliente($idCliente);
    $licenciasCliente = $this->getOnlyClienteLicencias($idCliente);

    # Dar formato a las cotizaciones
    $cotizaciones = [];
    foreach ($cotizacionesCliente as $row) {
      $folio = array("data" => $row['idCotizacion'],
        "class" => array("cursor-pointer", "vista_cotizacion_panel_ventas"));
      $fecha = array("data" => date('d/m/y', $row['createdAt']),
        "class" => array("cursor-pointer", "vista_cotizacion_panel_ventas"));
      $usuario = array("data" => user_load($row['createdBy'])->name,
        "class" => array("cursor-pointer", "vista_cotizacion_panel_ventas"));
      $desc = array("data" => mb_substr($row['__unkownProperties']['desc_cotizacion'], 0, 50, "utf-8"),
        "class" => array("cursor-pointer", "vista_cotizacion_panel_ventas"));
      $precio = array("data" => number_format($row['__unkownProperties']['subtotal'], 2, '.', ','),
        "class" => array("cursor-pointer", "vista_cotizacion_panel_ventas", "text-right"));
      $cotizaciones[] = array(
        "data" => array(
          $folio,
          $fecha,
          $usuario,
          $desc,
          $precio
        ),
        "data-id-cotizacion" => $row['idCotizacion']
      );
    }

    // Header -- Sort
    $header_cols = array(
      array("data" => "Folio", "class" => "filter-text"),
      array("data" => "Fecha", "class" => "filter-date"),
      array("data" => "Usuario", "class" => "filter-text"),
      array("data" => "Descripción", "class" => "filter-text"),
      array("data" => "Total", "class" => "filter-number")
    );

    $table = array(
      "header" => $header_cols,
      "rows" => $cotizaciones,
      "attributes" => array("id" => "table_reporteador_cotizaciones_cliente", "class" => array("table-sm"))
    );
    $table_cotizaciones = theme('table', $table);


    # Dar formato a las renovaciones.
    $renovaciones = [];
    foreach ($renovacionesCliente as $row) {
      $fecha_r = date('d/m/y', $row['fechaRenovacion']);
      $tipo_renovacion = $row['__unkownProperties']['nombre_tipo_renovacion'];
      $producto = $row['productoServicio'];
      $precio = array("data" => number_format($row['precio'], 2, '.', ','), "class" => array("text-right"));

      $renovaciones[] = array(
        "data" => array(
          $fecha_r,
          $tipo_renovacion,
          $producto,
          $precio
        ),
        "data-id-renovacion" => $row['idRenovacion']
      );
    }

    $header_renovaciones_cols = array(
      array("data" => "Fecha", "class" => "filter-date"),
      array("data" => "Tipo", "class" => "filter-text"),
      array("data" => "Producto", "class" => "filter-text"),
      array("data" => "Precio", "class" => "filter-number text-right")
    );

    $table_renovaciones = array(
      "header" => $header_renovaciones_cols,
      "rows" => $renovaciones,
      "sticky" => false,
      "attributes" => array("id" => "table_reporteador_renovaciones", "class" => array("table-sm"))
    );

    $result_renovaciones = theme('table', $table_renovaciones);


    // Licencias
    $licencias = [];
    foreach ($licenciasCliente as $row) {
      $fecha_l = date('d/m/y', $row['fechaCompraLicencia']);
      $licencia = $row['producto'];
      $serie_licencia = $row['serieLicencia'];
      $eliminar = array("data" => '<i class="fas fa-trash"></i>', "class" => array("text-center", "eliminar_licencia", "text-danger", "cursor-pointer"));
      $obs = array("data" => "Obs", "class" => array("observaciones_licencia", ($row['observaciones']) ? "status-yellow" : "", "cursor-pointer", "text-center"));
      $licencias[] = array(
        "data" => array(
          $fecha_l,
          $licencia,
          $serie_licencia,
          $obs,
          $eliminar
        ),
        "data-id-licencia" => $row['idLicencia'],
        "data-id-cliente" => $row['idCliente']
      );
    }
    $header_lic_cols = array(
      array("name" => array("data" => "Fecha Compra", "class" => "filter-date"), "sortable" => 0),
      array("name" => array("data" => "Producto", "class" => "filter-text"), "sortable" => 0),
      array("name" => array("data" => "Serie", "class" => "filter-text"), "sortable" => 0),
      array("name" => array("data" => "Obs", "class" => "filter-text text-center"), "sortable" => 0),
      array("name" => array("data" => '<i class="fas fa-trash"></i>', "class" => "filter-text text-center"), "sortable" => 0),
    );

    $header_lic = sys_tools::sort_table('id_licencia', 'DESC', $header_lic_cols);
    $table_lic = array(
      "header" => $header_lic,
      "rows" => $licencias,
      "sticky" => false,
      "attributes" => array("id" => "table_reporteador_licencias", "class" => array("table-sm"))
    );
    $result_lic = theme('table', $table_lic);
    $data['cliente'] = $Cliente->__toArray();
    $data['table_cotizaciones'] = $table_cotizaciones;
    $data['table_renovaciones'] = $result_renovaciones;
    $data['table_licencias'] = (empty($licencias)) ? '<p class="lead text-danger">El Cliente no cuenta con ninguna licencia.</p>' : $result_lic;

    # Este try-catch en teoria no seria necesario aqui, porque cuando se haga la de clientesRounter
    # en esa clase creo que si lo deberá de agarrar con su respectivo try-catch.
    $html = GenerateTemplate::Generate('clientes.xd', $data);

    return $html;
  }
}