<?php
/**
 * Archivo del sistema interno de pcpartners.com.mx.
 * User: Erick
 * Date: 17/05/2019
 * Time: 02:21 PM
 */

namespace Partners\Backups;


use Exception;
use Partners\Utils\GenerateTemplate;
use Partners\Utils\JsonMessageHelper;
use Partners\Utils\PartnersBitacora;
use Partners\Utils\PartnersException;
use Partners\Utils\PartnersLogger;
use Partners\Utils\PartnersTelegramBot;
use PDO;
use Swift_Mailer;
use Swift_Message;
use Swift_SendmailTransport;

class BackupsController
{
  private $user;

  public function __construct()
  {
    global $user;
    $this->user = $user;
  }

  /**
   * Método para mostrar un reporte de todos los registros de backups existentes.
   * @param $data
   * @throws Exception
   */
  public function reporteBase($data)
  {
    global $user;
    $query = db_select('partners_backups', 'pb');
    $query->fields('pb');
    $query->innerJoin('partners_backups_equipos', 'pbe', 'pbe.id_backup = pb.id_backup');
    $query->innerJoin('partners_clientes', 'pc', 'pb.id_cliente = pc.id_cliente');
    $query->addField('pc', 'nombre_cliente');
    $query->addExpression('COUNT(pbe.id_equipo_backup)', 'total_equipos');
    $query->groupBy('pb.id_backup');
    $query->condition('pb.eliminar', 2, '<>');

    if (!user_access('backups_administrador')) {
      $query->condition('pb.created_by', $user->uid);
    }

    $query->orderBy('id_backup', 'ASC');

    $backups = [];
    $query_datos = $query->execute();
    while ($row = $query_datos->fetchAssoc()) {
      $Backup = new Backup($row);
      switch ($Backup->getPaqueteRespaldo()) {
        case 1:
          $paquete = array("data" => '300 GB', "class" => "cursor-pointer cambiar_paquete_backup");
          $totalP = 300000;
          break;
        case 2:
          $paquete = array("data" => '500 GB', "class" => "cursor-pointer cambiar_paquete_backup");
          $totalP = 500000;
          break;
        case 3:
          $paquete = array("data" => '1 TB', "class" => "cursor-pointer cambiar_paquete_backup");
          $totalP = 1000000;
          break;
        case 4:
          $paquete = array("data" => '2 TB', "class" => "cursor-pointer cambiar_paquete_backup");
          $totalP = 2000000;
          break;
        default:
          $paquete = '';
          $totalP = 0;
      }


      $percentageUsed = ($Backup->getUsoReal() * 100) / $totalP;

      if ($percentageUsed >= 0 && $percentageUsed <= 50) {
        $color = 'bg-success';
      } else if ($percentageUsed > 50 && $percentageUsed <= 80) {
        $color = 'bg-warning';
      } else if ($percentageUsed > 80) {
        $color = 'bg-danger';
      } else {
        $color = '';
      }

      $finalPercentage = $percentageUsed > 100 ? 100 : $percentageUsed;

      if ($Backup->getUsoReal()) {
        $usoReal = array("data" => number_format($Backup->getUsoReal() / 1000, 2, '.', ',') . 'GB',
          "class" => array("progress-bar progress-bar-striped text-dark", $color),
          "style" => "width: $finalPercentage%");
        $fechaUsoReal = array("data" => date('d/m/Y H:i', $Backup->getUltimaFechaUsoReal()),
          "class" => array("text-center", "consultar_uso_bucket"));

      } else {
        $usoReal = '';
        $fechaUsoReal = array("data" => 'Sin consultar', "class" => array("text-center", "consultar_uso_bucket", "cursor-pointer"));
      }

      $bgTr = '';

      if ($Backup->getAccessKeyStatus() === 1) {
        $statusAccessKey = array("data" => '<i class="fas fa-user-check"></i>', "class" => "inactivar_access_key cursor-pointer text-center");
      } else {
        $bgTr = 'bg-warning';
        $statusAccessKey = array("data" => '<i class="fas fa-user-slash"></i>', "class" => "activar_access_key cursor-pointer text-center");
      }

      if ($Backup->getEliminar() === 0) {
        $eliminar = array("data" => '<i class="fas fa-trash"></i>', "class" => "cursor-pointer text-center pre_eliminar_backup");
      } else {
        $bgTr = 'bg-danger';
        $eliminar = array("data" => '<i class="fas fa-trash"></i>', "class" => "cursor-pointer text-center eliminar_backup text-white");
      }

      # Observaciones
      if ($Backup->getObservaciones()) {
        $obs = array("data" => "Obs", "class" => array("status-yellow", "text-center", "cursor-pointer", "status_obs"));
      } else {
        $obs = array("data" => "Obs", "class" => array("text-center", "cursor-pointer", "status_obs"));
      }

      $enviarSecretKey = array("data" => '<i class="fas fa-key"></i>', "class" => "text-center cursor-pointer send-secret-key");

      if (user_access('backups_administrador')) {
        $backups[] = array("data" => array(
          $Backup->getIdBackup(),
          user_load($Backup->getCreatedBy())->name,
          $Backup->getUnkownProperties()['nombre_cliente'],
          $Backup->getNombreBucket(),
          $paquete,
          array("data" => $Backup->getUnkownProperties()['total_equipos'],
            "class" => array("cursor-pointer", "text-center", "vista_equipos")),
          $Backup->getAccessKeyAWS(),
          $Backup->getSecretKeyAWS(),
          $usoReal,
          $fechaUsoReal,
          $obs,
          $statusAccessKey,
          $enviarSecretKey,
          $eliminar,
        ),
          "data-id-backup" => $Backup->getIdBackup(),
          "data-nombre-bucket" => $Backup->getNombreBucket(),
          "class" => $bgTr
        );
      } else {
        $backups[] = array("data" => array(
          $Backup->getIdBackup(),
          user_load($Backup->getCreatedBy())->name,
          $Backup->getUnkownProperties()['nombre_cliente'],
          $Backup->getNombreBucket(),
          $paquete,
          array("data" => $Backup->getUnkownProperties()['total_equipos'],
            "class" => array("cursor-pointer", "text-center", "vista_equipos")),
          $Backup->getAccessKeyAWS(),
          $usoReal,
          $fechaUsoReal,
          $obs
        ),
          "data-id-backup" => $Backup->getIdBackup(),
          "data-nombre-bucket" => $Backup->getNombreBucket(),
          "class" => $bgTr
        );
      }

    }

    if (user_access('backups_administrador')) {
      $header_cols = array(
        array("data" => "ID Backup"),
        array("data" => "Usuario"),
        array("data" => "Nombre Cliente"),
        array("data" => "Nombre Bucket"),
        array("data" => "Paquete"),
        array("data" => "Equipos"),
        array("data" => "Access Key"),
        array("data" => "Secret Key"),
        array("data" => "Uso Real"),
        array("data" => "Última Consulta", "class" => array("text-center")),
        array("data" => "Obs"),
        array("data" => '<i class="fas fa-user"></i>', "class" => "text-center"),
        array("data" => '<i class="fas fa-key"></i>', "class" => "text-center"),
        array("data" => '<i class="fas fa-trash"></i>', "class" => "text-center"),
      );
    } else {
      $header_cols = array(
        array("data" => "ID Backup"),
        array("data" => "Usuario"),
        array("data" => "Nombre Cliente"),
        array("data" => "Nombre Bucket"),
        array("data" => "Paquete"),
        array("data" => "Equipos"),
        array("data" => "Access Key"),
        array("data" => "Uso Real"),
        array("data" => "Última Consulta", "class" => array("text-center")),
        array("data" => "Obs")
      );
    }


    $table = array(
      "header" => $header_cols,
      "rows" => $backups,
      "sticky" => false,
      "attributes" => array("id" => "table_reporteador_backups", "class" => array("table-sm"))
    );

    $result = theme('table', $table);

    JsonMessageHelper::Success($result);
  }

  /**
   * Método que revisa una serie de reglas para validar que los buckets no excedan su límite de espacio
   */
  public function sendEmailsBackupsStatus()
  {
    $getInfo = db_select('partners_backups', 'pb');
    $getInfo->fields('pb', array('nombre_bucket', 'id_backup', 'paquete_respaldo', 'access_key_aws', 'uso_real'));
    $getInfo->innerJoin('partners_clientes', 'pc', 'pc.id_cliente = pb.id_cliente');
    $getInfo->addField('pc', 'nombre_cliente');
    $getInfo->leftJoin('partners_backups_equipos', 'pbe', 'pbe.id_backup = pb.id_backup');
    $getInfo->addExpression('count(pbe.id_equipo_backup)', 'equipos');
    $getInfo->groupBy('pb.id_backup');
    $buckets = $getInfo->execute()->fetchAll(PDO::FETCH_ASSOC);
    foreach ($buckets as $bucket) {
      $Backup = new Backup($bucket);
      switch ($Backup->getPaqueteRespaldo()) {
        case 1:
          $paquete = '300 GB';
          $totalP = 300000;
          break;
        case 2:
          $paquete = '500 GB';
          $totalP = 500000;
          break;
        case 3:
          $paquete = '1 TB';
          $totalP = 1000000;
          break;
        case 4:
          $paquete = '2 TB';
          $totalP = 2000000;
          break;
        default:
          $totalP = 0;
          $paquete = '';
      }

      $percentageUsed = ($Backup->getUsoReal() * 100) / $totalP;


      $data['bucket_name'] = $Backup->getNombreBucket();
      $data['percentage'] = number_format($percentageUsed, 2, '.', '');
      $data['nombre_cliente'] = $Backup->getUnkownProperties()['nombre_cliente'];
      $data['paquete'] = $paquete;
      $data['uso_real'] = number_format($Backup->getUsoReal() / 1000, 2, '.', '');
      $data['equipos'] = $Backup->getUnkownProperties()['equipos'];
      $sendMail = false;
      $body = '';
      $subject = '';

      # Si 90 < uso total < 100
      if ($percentageUsed >= 90 && $percentageUsed < 100) {
        $body = render_template('php', 'backups.mail_warning_bucket_usage', $data);
        $subject = 'Pc Partners Respaldos - Un respaldo está llegando a su capacidad máxima.';
        $sendMail = true;
      } else if ($percentageUsed >= 100) {
        // Cancelar Access Key
        $Aws = new AwsClient();
        try {
          $r = $Aws->changeStatusAccessKey($Backup->getAccessKeyAWS(), $Backup->getNombreBucket(), 0);
          if ($r) {
            # Inactivar la access key de AWS.
            db_update('partners_backups')
              ->fields(array(
                'access_key_status' => 0
              ))
              ->condition('id_backup', $Backup->getIdBackup())
              ->execute();
            PartnersLogger::Info('partners_backups', 'update',
              "Se actualizó el status del access key (0) al servicio de respaldo " . $Backup->getIdBackup());

            $body = render_template('php', 'backups.mail_error_bucket_usage', $data);
            $subject = 'Pc Partners Respaldos - Un respaldo excedió el límite de su capacidad.';
            $sendMail = true;
          }
        } catch (PartnersException $e) {
          PartnersLogger::Error($e, 'backupsControlller.sendEmailsBackupsStatus');
        }
      }

      // Enviar correo
      if ($sendMail) {
        $transport = new Swift_SendmailTransport();
        $mailer = new Swift_Mailer($transport);
        $from = ['respaldos-noreply@pcpartners.com.mx' => 'Pc Partners - Respaldos'];

        $to = [
          'roberto@pcpartners.com.mx' => 'Pc Partners - Roberto'
        ];
        $message = (new Swift_Message())
          ->setSubject($subject)
          ->setFrom($from)
          ->setBody($body, 'text/html');
        $failedRecipients = [];
        $numSent = 0;

        foreach ($to as $address => $name) {
          if (is_int($address)) {
            if (!empty($message)) {
              $message->setTo($name);
            }
          } else {
            if (!empty($message)) {
              $message->setTo([$address => $name]);
            }
          }
          PartnersLogger::Info('partners_backups', 'mail',
            "Se envió automáticamente un correo de aviso de respaldos: $subject.");
          if (!empty($message)) {
            $numSent += $mailer->send($message, $failedRecipients);
          }
        }
      }
    }
  }

  /**
   * Método para generar un formulario para capturar un nuevo servicio
   * de respaldo.
   * @throws PartnersException
   */
  public function vistaNuevoBackup()
  {
    $html = GenerateTemplate::Generate('backups.nuevo_backup');
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para generar un servicio de backup, creando toda la información
   * necesaria en Wasabi y AWS:
   * @param $datos
   */
  public function capturarBackup($datos)
  {
    $Backup = new Backup([
      'nombre_bucket' => $datos->nombre_bucket,
      'paquete_respaldo' => $datos->paquete_respaldo,
      'id_cliente' => $datos->id_cliente,
      'id_contacto' => $datos->id_contacto]);
    $Backup->setListEquipos((array)$datos->listEquipos);

    // Capturar Renovación y Pendiente
    $token = substr(str_shuffle("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"), 0, 10);

    try {
      $i = 0;
      $productoServicio = 'Servicio de Respaldo PartnersBackups en Wasabi';
      $Bitacora = new PartnersBitacora($datos->observaciones_renovacion);
      foreach ($datos->dates as $date) {
        $i++;
        db_insert('partners_renovaciones')
          ->fields(array(
            'created_at' => REQUEST_TIME,
            'created_by' => $this->user->uid,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $this->user->uid,
            'id_cliente' => $Backup->getIdCliente(),
            'id_contacto' => $Backup->getIdContacto() === 0 ? null : $Backup->getIdContacto(),
            'id_tipo' => 133, // PartnersBackups
            'id_area_renovacion' => 2, // Area Soporte
            'producto_servicio' => $productoServicio,
            'detalle' => '',
            'fecha_inicio' => $datos->fecha_renovacion,
            'fecha_renovacion' => $date,
            'frecuencia_intervalo' => $datos->input_frecuencia, // 1,2,3...
            'frecuencia_tipo' => $datos->select_frecuencia, // YEARLY; WEEKLY, DAILY
            'precio' => $datos->precio_renovacion,
            'observaciones' => $Bitacora->generarNuevaObservacion(),
            'token_relacion_renovacion' => $token . $i
          ))
          ->execute();

        db_insert('partners_pendientes')
          ->fields(array(
            'created_at' => REQUEST_TIME,
            'created_by' => $this->user->uid,
            'updated_at' => REQUEST_TIME,
            'updated_by' => $this->user->uid,
            'tipo_pendiente' => 3, // Renovación
            'area_pendiente' => 2, // Area Soporte
            'id_cliente' => $datos->id_cliente,
            'asignado_para' => 0,
            'fecha_pendiente' => $date,
            'descripcion_pendiente' => "Renovación de PartnersBackups: $productoServicio",
            'status' => 1,
            'horario_pendiente' => '',
            'prioridad_pendiente' => 5,
            'que_llevar_pendiente' => '',
            'token_relacion_renovacion' => $token . $i
          ))
          ->execute();
      }

      PartnersLogger::Info('partners_renovaciones', 'insert',
        'Se capaturó una renovación de un servicio de respaldo');

      PartnersLogger::Info('partners_pendientes', 'insert',
        'Se capturó un pendiente de una renovación de un servicio de respaldo');
    } catch (Exception $e) {
      JsonMessageHelper::Error($e->getMessage());
    }


    // Crear cuenta y bucket en AWS
    try {
      $AwsClient = new AwsClient();
      $userName = $datos->nombre_bucket;

      $ApiKeys = $AwsClient->crearUsuarioAWS($userName);
      $policyArn = $AwsClient->crearPoliticaAWS($userName);
      $AwsClient->asignarPoliticaUsuario($userName, $policyArn);
      $s3BucketLocation = $AwsClient->crearBucketAWS($userName);

      $Backup->setAccessKeyAWS($ApiKeys['accessKey']);
      $Backup->setSecretKeyAWS($ApiKeys['secretKey']);
      $Backup->setUnkownProperties(['s3BucketLocation' => $s3BucketLocation]);

      PartnersLogger::Info('wasabi_aws', 'service',
        "Se creó un usuario con su respectivo bucket: $userName");
    } catch (PartnersException $e) {
      JsonMessageHelper::Error($e->getMessage());
    }

    try {
      // Capturar registro Backup
      $idBackup = db_insert('partners_backups')
        ->fields(array(
          'created_at' => REQUEST_TIME,
          'created_by' => $this->user->uid,
          'updated_at' => REQUEST_TIME,
          'updated_by' => $this->user->uid,
          'id_cliente' => $Backup->getIdCliente(),
          'id_contacto' => $Backup->getIdContacto(),
          'nombre_bucket' => $Backup->getNombreBucket(),
          'paquete_respaldo' => $Backup->getPaqueteRespaldo(),
          'token_renovacion' => $token,
          'access_key_aws' => $Backup->getAccessKeyAWS(),
          'secret_key_aws' => $Backup->getSecretKeyAWS()
        ))
        ->execute();

      PartnersLogger::Info('partners_backups', 'insert',
        "Se dió de alta un servicio de backup, id_backup: $idBackup");

      // Capturar Equipos
      foreach ($Backup->getListEquipos() as $equipo) {
        db_insert('partners_backups_equipos')
          ->fields(array(
            'id_backup' => $idBackup,
            'nombre_equipo' => $equipo->nombre_equipo,
            'uso_espacio' => $equipo->uso_espacio * 1000 // Para que se capturen como MB
          ))
          ->execute();
      }

      JsonMessageHelper::Success($idBackup);
    } catch (Exception $e) {
      JsonMessageHelper::Error($e->getMessage());
    }
  }

  /**
   * Método para generar una vista con los equipos de un servicio de respaldo.
   * @param $datos
   */
  public function vistaEquipos($datos)
  {
    $idBackup = $datos['id_backup'];

    $q = db_select('partners_backups_equipos', 'pbe');
    $q->fields('pbe');
    $q->condition('id_backup', $idBackup);

    /** @noinspection PhpUndefinedMethodInspection */
    $dataE = $q->execute()->fetchAll(PDO::FETCH_ASSOC);

    $qN = db_select('partners_backups', 'pb');
    $qN->fields('pb', array('nombre_bucket', 'paquete_respaldo', ' id_backup'));
    $qN->condition('id_backup', $idBackup);
    $datosBucket = $qN->execute()->fetchAssoc();

    $data['nombre_bucket'] = $datosBucket['nombre_bucket'];
    $data['id_backup'] = $datosBucket['id_backup'];
    $data['paquete'] = $datosBucket['paquete_respaldo'];
    $equipos = [];
    foreach ($dataE as $item) {

      $percentageUsed = ($item['uso_real'] * 100) / $item['uso_espacio'];

      if ($percentageUsed >= 0 && $percentageUsed <= 50) {
        $item['color'] = 'bg-success';
      } else if ($percentageUsed > 50 && $percentageUsed <= 80) {
        $item['color'] = 'bg-warning';
      } else if ($percentageUsed > 80) {
        $item['color'] = 'bg-danger';
      }

      $item['disk_used'] = $percentageUsed > 100 ? 100 : $percentageUsed;
      $item['uso_real'] = $item['uso_real'] != null ? $item['uso_real'] / 1000 : 0;
      $item['uso_espacio'] = $item['uso_espacio'] / 1000;
      $equipos[] = $item;
    }

    $data['equipos'] = $equipos;
    $html = '';
    try {
      $html = GenerateTemplate::Generate('backups.vista_equipos', $data);
    } catch (PartnersException $e) {
      JsonMessageHelper::Error($e->getMessage());
    }
    JsonMessageHelper::Success($html);
  }

  /**
   * Método para consultar el tamaño de un equipo (folder) en AWS.
   * @param $datos
   */
  public function consultarEspacioEquipo($datos)
  {
    $getInfo = db_select('partners_backups_equipos', 'pbe');
    $getInfo->addField('pbe', 'nombre_equipo');
    $getInfo->addField('pb', 'nombre_bucket');
    $getInfo->innerJoin('partners_backups', 'pb', 'pb.id_backup = pbe.id_backup');
    $getInfo->condition('pbe.id_equipo_backup', $datos['id_equipo']);
    $info = $getInfo->execute()->fetchAssoc();
    $nombreBucket = $info['nombre_bucket'] . '-Bucket'; # Se agrega el postfix de -Bucket porque así quedó en AWS

    # Se agrega el prefix de CBB que es el que pone cloudberry.
    $nombreEquipo = 'CBB_' . $info['nombre_equipo'];
    $AwsClient = new AwsClient();
    $totalSizeMB = $AwsClient->getFolderSize($nombreBucket, $nombreEquipo);

    if ($totalSizeMB == -1) {
      JsonMessageHelper::Error("No se encontró ninguna carpeta con el nombre de $nombreEquipo o 
      ningún bucket con el nombre de /$nombreBucket");
    } else {
      # Actualizar el equipo con el último tamaño calculado
      db_update('partners_backups_equipos')
        ->fields(array(
          'uso_real' => $totalSizeMB,
          'ultima_fecha_uso_real' => REQUEST_TIME
        ))
        ->condition('id_equipo_backup', $datos['id_equipo'])
        ->execute();

      PartnersLogger::Info('partners_backups_equipos', 'update',
        "Se consultó el tamaño de un equipo en AWS, idEquipo:" . $datos['id_equipo']);
      JsonMessageHelper::Success($totalSizeMB);
    }
  }

  /**
   * Método para consultar el tamaño de un bucket en AWS.
   * @param $datos
   */
  public function consultarEspacioBucket($datos)
  {
    $getInfo = db_select('partners_backups', 'pb');
    $getInfo->addField('pb', 'nombre_bucket');
    $getInfo->condition('id_backup', $datos['id_backup']);
    $nombreBucket = $getInfo->execute()->fetchField() . '-Bucket'; # Se agrega el postfix de -Bucket porque así quedó en AWS

    # Se agrega el prefix de CBB que es el que pone cloudberry.
    $AwsClient = new AwsClient();
    $totalSizeMB = $AwsClient->getBucketSize($nombreBucket);

    if ($totalSizeMB == -1) {
      JsonMessageHelper::Error("No se encontró ningún bucket con el nombre de /$nombreBucket");
    } else {
      # Actualizar el equipo con el último tamaño calculado
      db_update('partners_backups')
        ->fields(array(
          'uso_real' => $totalSizeMB,
          'ultima_fecha_uso_real' => REQUEST_TIME
        ))
        ->condition('id_backup', $datos['id_backup'])
        ->execute();

      PartnersLogger::Info('partners_backups', 'update',
        "Se consultó el tamaño de un bucket en AWS, idBucket:" . $datos['id_backup']);
      JsonMessageHelper::Success($totalSizeMB);
    }
  }

  /**
   * Método para capturar todos los cambios realizados a los equipos de un servicio
   * de respaldo.
   * @param $datos
   */
  public function capturarEdicionEquipos($datos)
  {
    foreach ($datos->listEquipos as $equipo) {
      if ($equipo->deleted) {
        // Eliminar equipo
        db_delete('partners_backups_equipos')
          ->condition('id_equipo_backup', $equipo->idEquipo)
          ->execute();
      } else if ($equipo->newEquipo) {
        // Nuevo equipo
        try {
          db_insert('partners_backups_equipos')
            ->fields(array(
              'id_backup' => $datos->idBackup,
              'nombre_equipo' => $equipo->nombreEquipo,
              'uso_espacio' => $equipo->espacioEquipo * 1000 // Se guarda en MB
            ))
            ->execute();
        } catch (Exception $e) {
          JsonMessageHelper::Error('Ocurrió un error al capturar un nuevo equipo');
          break;
        }
      } else {
        // Actualizar equipo existente
        db_update('partners_backups_equipos')
          ->fields(array(
            'nombre_equipo' => $equipo->nombreEquipo,
            'uso_espacio' => $equipo->espacioEquipo * 1000 // Se guarda en MB
          ))
          ->condition('id_equipo_backup', $equipo->idEquipo)
          ->execute();
      }
    }
    try {
      $Bitacora = new PartnersBitacora("Se hicieron modificaciones a los equipos.",
        null, null, 'partners_backups', 'observaciones', 'id_backup', $datos->idBackup);
      $Bitacora->capturarObs();
    } catch (Exception $e) {
      JsonMessageHelper::Error($e->getMessage());
    }
    JsonMessageHelper::Success('Datos actualizados correctamente!');
  }

  /**
   * Método para modificar el paquete de un servicio de respaldo.
   * @param $datos
   */
  public function capturarEdicionPaquete($datos)
  {
    db_update('partners_backups')
      ->fields(array(
        'paquete_respaldo' => $datos->paquete
      ))
      ->condition('id_backup', $datos->idBackup)
      ->execute();
    PartnersLogger::Info('partners_backups', 'update', "Se cambió el paquete ($datos->paquete) del backup $datos->idBackup");
    try {
      $Bitacora = new PartnersBitacora("Se modificó el paquete a $datos->paquete (1: 300GB, 2:500GB, 3:1TB, 4:2TB)",
        null, null, 'partners_backups', 'observaciones', 'id_backup', $datos->idBackup);
      $Bitacora->capturarObs();
    } catch (Exception $e) {
      JsonMessageHelper::Error($e->getMessage());
    }
    JsonMessageHelper::Success('Paquete modificado correctamente!');
  }

  /**
   * Método para marcar un registro para que sea eliminado después.
   * @param $datos
   */
  public function preEliminarBackup($datos)
  {
    $idBackup = $datos->idBackup;
    db_update('partners_backups')
      ->fields(array(
        'eliminar' => 1 // Pre eliminar
      ))
      ->condition('id_backup', $idBackup)
      ->execute();
    PartnersLogger::Info('partners_backups', 'update', "Se marcó para eliminar el backup $idBackup");
    try {
      $Bitacora = new PartnersBitacora("Se marcó para eliminar en un futuro.",
        null, null, 'partners_backups', 'observaciones', 'id_backup', $datos->idBackup);
      $Bitacora->capturarObs();
    } catch (Exception $e) {
      JsonMessageHelper::Error($e->getMessage());
    }
    JsonMessageHelper::Success('Marcado correctamente!');
  }

  /**
   * Método para eliminar un registro permanentemente.
   * @param $datos
   */
  public function eliminarBackup($datos)
  {
    $idBackup = $datos->idBackup;
    db_update('partners_backups')
      ->fields(array(
        'eliminar' => 2 // Eliminado
      ))
      ->condition('id_backup', $idBackup)
      ->execute();
    PartnersLogger::Info('partners_backups', 'delete', "Se eliminó el servicio de backup $idBackup");
    try {
      $Bitacora = new PartnersBitacora("Se eliminó el servicio de backup $datos->idBackup",
        null, null, 'partners_backups', 'observaciones', 'id_backup', $datos->idBackup);
      $Bitacora->capturarObs();
    } catch (Exception $e) {
      JsonMessageHelper::Error($e->getMessage());
    }
    JsonMessageHelper::Success('Respaldo eliminado correctamente!');
  }

  /**
   * Método para cambiar el status de una access key en AWS.
   * @param $datos
   */
  public function cambiarStatusAccessKey($datos)
  {
    $getD = db_select('partners_backups', 'pb');
    $getD->fields('pb', array('nombre_bucket', 'access_key_aws'));
    $getD->condition('id_backup', $datos->idBackup);
    $d = $getD->execute()->fetchAssoc();
    try {
      $Aws = new AwsClient();
      $r = $Aws->changeStatusAccessKey($d['access_key_aws'], $d['nombre_bucket'], $datos->status);
      if ($r) {
        db_update('partners_backups')
          ->fields(array(
            'access_key_status' => $datos->status
          ))
          ->condition('id_backup', $datos->idBackup)
          ->execute();
        PartnersLogger::Info('partners_backups', 'update',
          "Se actualizó el status del access key ($datos->status) al servicio de respaldo $datos->idBackup");
        try {
          $Bitacora = new PartnersBitacora("Se actualizó el status de la llave de acceso a $datos->status (1: Activo / 0: Inactivo)",
            null, null, 'partners_backups', 'observaciones', 'id_backup', $datos->idBackup);
          $Bitacora->capturarObs();
        } catch (Exception $e) {
          JsonMessageHelper::Error($e->getMessage());
        }
        JsonMessageHelper::Success('Status actualizado correctamente.');
      } else {
        JsonMessageHelper::Error('Ocurrió un error al actualizar el status de la Access Key');
      }
    } catch (PartnersException $e) {
      JsonMessageHelper::Error($e->getMessage());
    }
  }

  /**
   * Método para controlar las observaciones de los backups.
   * @param $datos
   * @throws Exception
   */
  public function loadObs($datos)
  {
    $Bitacora = new PartnersBitacora();
    $Bitacora->setTableName($datos->tabla);
    $Bitacora->setColName($datos->col);
    $Bitacora->setId($datos->idIndex);
    $Bitacora->setPrimaryKey($datos->index);

    // Consultar las observaciones
    if ($datos->type === 1) {
      $obs = $Bitacora->loadObs();
      JsonMessageHelper::Success($obs);
    } else if ($datos->type === 2) {
      // Capturar una observacion
      $Bitacora->setTextoObservacion($datos->new_val);
      $newObs = $Bitacora->capturarObs();

      JsonMessageHelper::Success($newObs);
    }
  }

  /**
   * Método para actualizar la información del espacio usado que tienen todos los buckets.
   */
  public function autoUpdateBucketSizeInfo()
  {
    $getInfo = db_select('partners_backups', 'pb');
    $getInfo->fields('pb', array('nombre_bucket', 'id_backup'));
    $buckets = $getInfo->execute()->fetchAll(PDO::FETCH_ASSOC);

    foreach ($buckets as $bucket) {
      $nombreBucket = $bucket['nombre_bucket'] . '-Bucket'; # Se agrega el postfix de -Bucket porque así quedó en AWS
      $AwsClient = new AwsClient();
      $totalSizeMB = $AwsClient->getBucketSize($nombreBucket);

      if ($totalSizeMB > -1) {
        # Actualizar el equipo con el último tamaño calculado
        db_update('partners_backups')
          ->fields(array(
            'uso_real' => $totalSizeMB,
            'ultima_fecha_uso_real' => REQUEST_TIME
          ))
          ->condition('id_backup', $bucket['id_backup'])
          ->execute();

        PartnersLogger::Info('partners_backups', 'update',
          "Se consultó el tamaño de un bucket en AWS, idBucket:" . $bucket['id_backup']);
      }
    }
  }

  /**
   * Método para generar una vista de la ficha de un backup.
   * @param $datos
   */
  public function vistaFichaBackup($datos)
  {
    $idBackup = $datos['id_backup'];
    $getInfo = db_select('partners_backups', 'pb');
    $getInfo->fields('pb', array('nombre_bucket', 'id_backup', 'paquete_respaldo', 'access_key_aws', 'secret_key_aws'));
    $getInfo->innerJoin('partners_clientes', 'pc', 'pc.id_cliente = pb.id_cliente');
    $getInfo->addField('pc', 'nombre_cliente');
    $getInfo->condition('id_backup', $idBackup);
    $bucket = $getInfo->execute()->fetchAssoc();

    $getEquipos = db_select('partners_backups_equipos', 'pbe');
    $getEquipos->fields('pbe', array('nombre_equipo', 'uso_espacio'));
    $getEquipos->condition('id_backup', $idBackup);
    $equipos = $getEquipos->execute()->fetchAll(PDO::FETCH_ASSOC);

    $data = [];

    foreach ($equipos as $equipo) {
      $data['equipos'][] = [
        'nombre_equipo' => $equipo['nombre_equipo'],
        'uso_espacio' => number_format($equipo['uso_espacio'] / 1000, 2, '.', '')
      ];
    }

    switch ($bucket['paquete_respaldo']) {
      case 1:
        $paquete = '300 GB';
        break;
      case 2:
        $paquete = '500 GB';
        break;
      case 3:
        $paquete = '1 TB';
        break;
      case 4:
        $paquete = '2 TB';
        break;
      default:
        $paquete = '';
    }

    $bucket['paquete_respaldo'] = $paquete;
    $data['bucket'] = $bucket;
    try {
      $html = GenerateTemplate::Generate('backups.ficha_informacion', $data);
      JsonMessageHelper::Success($html);
    } catch (PartnersException $e) {
      JsonMessageHelper::Error('Ocurrió un error al generar la ficha.' . $e->getMessage());
    }
  }

  /**
   * Método para generar una lista de los técnicos (área soporte) en el sistema.
   */
  public function listaTecnicos()
  {
    $role = user_role_load_by_name('Soporte');
    $query = db_select('users_roles', 'ur');
    $query->addField('ur', 'uid');
    $query->condition('rid', $role->rid);
    $uids = $query->execute()->fetchCol();

    $getUsers = db_select('users', 'u')
      ->fields('u', array('uid', 'name'))
      ->condition('status', 1)
      ->condition('uid', $uids, 'IN')
      ->condition('uid', 26, '!=')
      ->orderBy('name', 'ASC')
      ->execute()->fetchAll(PDO::FETCH_ASSOC);

    $usuarios = [];
    foreach ($getUsers as $user) {
      $usuarios[$user['uid']] = $user['name'];
    }

    JsonMessageHelper::Success($usuarios);
  }

  /**
   * Método para enviar por telegram un Token a un usuario en especifico para que pueda consultar
   * la ficha de un respaldo.
   * @param $datos
   */
  public function sendTokenSecretKey($datos)
  {
    global $user;
    $token = mt_rand(100000, 999999);
    try {
      db_insert('partners_backups_tokens')
        ->fields(array(
          'id_backup' => $datos->idBackup,
          'created_at' => REQUEST_TIME,
          'created_by' => $user->uid,
          'uid_solicita' => $datos->uid,
          'token' => $token
        ))
        ->execute();

      $Bitacora = new PartnersBitacora("Se generó un token para el usuario " .
        user_load($datos->uid)->name . " para ver la ficha de backup.", null, null,
        'partners_backups', 'observaciones', 'id_backup', $datos->idBackup);
      $Bitacora->capturarObs();

      $chatId = PartnersTelegramBot::getChatId($datos->uid);
      $Bot = new PartnersTelegramBot("El código para ver la ficha del respaldo es: $token", $chatId);
      $send = $Bot->sendMessage();
      if ($send) {
        JsonMessageHelper::Success('Token enviado correctamente');
      } else {
        JsonMessageHelper::Error('Ocurrió un error al enviar el mensaje.');
      }
    } catch (Exception $e) {
      JsonMessageHelper::Error('Ocurrió un error al capturar el token en la base de datos.');
    }
  }

  /**
   * Método para validar que un código sea valido, en caso de serlo, lo marca como ya usado y regresa
   * el id_backup para posteriormente generar la vista de la ficha de dicho respaldo.
   * @param $datos
   */
  public function validarCodigoFicha($datos)
  {
    $codigo = $datos->codigo;
    try {
      $db = db_select('partners_backups_tokens', 'pbt');
      $db->fields('pbt', ['id_backup', 'usado']);
      $db->condition('token', $codigo);
      $backup = $db->execute()->fetchAssoc();
      if ($backup) {
        if ($backup['usado'] == 0) {
          db_update('partners_backups_tokens')
            ->fields(array(
              'usado' => REQUEST_TIME
            ))
            ->condition('token', $codigo)
            ->execute();

          $Bitacora = new PartnersBitacora("Se usó el código: $codigo", null, 'Anónimo',
            'partners_backups', 'observaciones', 'id_backup', $backup['id_backup']);
          $Bitacora->capturarObs();

          JsonMessageHelper::Success($backup['id_backup']);
        } else {
          JsonMessageHelper::Error('El código ya ha sido usado, recuerda que los códigos solo tienen un solo uso!');
        }
      } else {
        JsonMessageHelper::Error('El código que usaste no existe');
      }
    } catch (Exception $e) {
      JsonMessageHelper::Error('Ocurrió un error al validar el código');
    }
  }
}