<?php
/**
 * Archivo del sistema interno de pcpartners.com.mx.
 * User: Erick
 * Date: 18/05/2019
 * Time: 01:21 PM
 */

namespace Partners\Backups;


use Aws\Credentials\CredentialProvider;
use Aws\Exception\AwsException;
use Aws\Iam\IamClient;
use Aws\S3\S3Client;
use Partners\Utils\JsonMessageHelper;
use Partners\Utils\PartnersException;

class AwsClient
{
  private $regionAWS = 'us-east-1';
  private $provider;
  private $s3Endpoint = 'https://s3.wasabisys.com';
  private $endpoint = 'https://wasabisys.com';
  private $IamClient;
  private $s3Client;

  /**
   * Constructor que inicializa las llaves necesarias para AWS.
   * AwsClient constructor.
   */
  public function __construct()
  {
    $provider = CredentialProvider::ini(null, HOME_SERVER . '.aws/credentials');
    $provider = CredentialProvider::memoize($provider);
    $this->provider = $provider;

    $this->IamClient = new IamClient([
      'version' => '2010-05-08',
      'region' => $this->regionAWS,
      'endpoint' => $this->endpoint,
      'credentials' => $this->provider
    ]);

    $this->s3Client = new S3Client([
      'region' => $this->regionAWS,
      'version' => '2006-03-01',
      'endpoint' => $this->s3Endpoint,
      'credentials' => $this->provider
    ]);
  }

  /**
   * Método para crear un usuario y generar sus respectivas API Keys
   * @param $userName string Nombre de usuario que se creará en IAM
   * @return array Regresa array con accessKey y secretKey generadas para el usuario recien creado.
   * @throws PartnersException
   */
  public function crearUsuarioAWS($userName)
  {
    try {
      $this->IamClient->createUser(['UserName' => $userName]);

      $resultCreateApiKeys = $this->IamClient->createAccessKey([
        'UserName' => $userName,
      ]);

      $accessKey = $resultCreateApiKeys['AccessKey']['AccessKeyId'];
      $secretKey = $resultCreateApiKeys['AccessKey']['SecretAccessKey'];

      return [
        'accessKey' => $accessKey,
        'secretKey' => $secretKey
      ];
    } catch (AwsException $e) {
      throw new PartnersException($e);
    }
  }

  /**
   * Método para crear una política especifica al usuario que se indica.
   * @param $prefixPolitica
   * @return string  Regresa el Arn de la política para que después se le agregue al usuario.
   * @throws PartnersException
   */
  public function crearPoliticaAWS($prefixPolitica)
  {
    try {
      $policyName = $prefixPolitica . 'PolicyApi';
      $resultCreatePolicy = $this->IamClient->createPolicy([
        'PolicyName' => $policyName,
        'PolicyDocument' => '{ "Id": "Policy' . time() . '", "Version": "2012-10-17", "Statement": [ { "Sid": "Stmt' . time() . '", "Effect": "Allow", "Principal": { "AWS": "arn:aws:iam::100000011500:user/' . $prefixPolitica . '" }, "Action": [ "s3:AbortMultipartUpload", "s3:PutObject", "s3:Get*", "s3:List*" ], "Resource": [ "arn:aws:s3:::' . $prefixPolitica . '-Bucket/*", "arn:aws:s3:::' . $prefixPolitica . '-Bucket" ] }, { "Sid": "Stmt' . time() . '", "Effect": "Allow", "Principal": { "AWS": "arn:aws:iam::100000011500:user/' . $prefixPolitica . '" }, "Action": "s3:listAllMyBuckets", "Resource": "arn:aws:s3:::*" } ] }'
      ]);

      return $resultCreatePolicy['Policy']['Arn'];
    } catch (AwsException $e) {
      throw new PartnersException($e);
    }
  }

  /**
   * Método para crear un bucket con un nombre en específico.
   * @param $prefix string Bajo que prefijo quedará el nombre del bucket
   * @return string Ruta del nuevo bucket creado
   * @throws PartnersException
   */
  public function crearBucketAWS($prefix)
  {
    try {
      $bucketName = $prefix . '-Bucket';
      $resultCreateBucket = $this->s3Client->createBucket([
        'Bucket' => $bucketName,
      ]);

      return $resultCreateBucket['Location'];
    } catch (AwsException $e) {
      throw new PartnersException($e);
    }
  }

  /**
   * Método para asignar una víncular una política con un usuario.
   * @param $userName
   * @param $policyArn
   * @throws PartnersException
   */
  public function asignarPoliticaUsuario($userName, $policyArn)
  {
    try {
      $this->IamClient->attachUserPolicy([
        'PolicyArn' => $policyArn,
        'UserName' => $userName
      ]);
    } catch (AwsException $e) {
      throw new PartnersException($e);
    }
  }

  /**
   * Método para consultar en AWS el tamaño de una carpeta en especifico.
   * @param $bucketName
   * @param $prefix
   * @return float
   */
  public function getFolderSize($bucketName, $prefix)
  {
    try {
      $r = $this->s3Client->getIterator('ListObjects', [
        'Bucket' => $bucketName,
        'Prefix' => $prefix,
      ]);
      $totalSize = 0;
      foreach ($r as $item) {
        $totalSize += $item['Size'];
      }
      return $totalSize == 0 ? -1 : $this->formatSizeUnits($totalSize);
    } catch (AwsException $e) {
      return -1;
    }
  }

  /**
   * En base a bytes, regresa la cantidad de MB
   * @param $bytes
   * @return float
   */
  public function formatSizeUnits($bytes)
  {
    $bytes = $bytes / 1048576;
    return $bytes;
  }

  /**
   * Método para consultar en AWS el tamaño de un bucket en especifico.
   * @param $bucketName
   * @return float
   */
  public function getBucketSize($bucketName)
  {
    try {
      $r = $this->s3Client->getIterator('ListObjects', [
        'Bucket' => $bucketName,
      ]);
      $totalSize = 0;
      foreach ($r as $item) {
        $totalSize += $item['Size'];
      }
      return $totalSize == 0 ? -1 : $this->formatSizeUnits($totalSize);
    } catch (AwsException $e) {
      return -1;
    }
  }

  /**
   * Método para cambiar el status de una llave de acceso en AWS.
   * @param $accessKey
   * @param $userName
   * @param $status
   * @return bool
   * @throws PartnersException
   */
  public function changeStatusAccessKey($accessKey, $userName, $status)
  {
    $statusName = '';
    if ($status === 1) {
      $statusName = 'Active';
    } else if ($status === 0) {
      $statusName = 'Inactive';
    }

    try {
      $this->IamClient->updateAccessKey([
        'AccessKeyId' => $accessKey,
        'Status' => $statusName,
        'UserName' => $userName,
      ]);
      return true;
    } catch (AwsException $e) {
      throw new PartnersException($e);
    }
  }
}